import sys, os, subprocess, xbmc, xbmcgui, xbmcaddon, xbmcvfs, socket
import shutil
from datetime import datetime

# --- Helper Functions (Keep your existing ones) ---

def get_paths():
    addon = xbmcaddon.Addon()
    base_path = xbmcvfs.translatePath(addon.getAddonInfo('path'))
    res = os.path.join(base_path, 'resources')
    
    for folder in ['certs', 'logs', 'scripts']:
        path = os.path.join(res, folder)
        if not os.path.exists(path): os.makedirs(path)

    return {
        "bin": "/usr/bin/openvpn",
        "res": res,
        "conf": os.path.join(res, 'server.conf'),
        "cnf": os.path.join(res, 'openssl.cnf'),
        "log": os.path.join(res, 'logs', 'openvpn.log'),
        "ca": os.path.join(res, 'certs', 'ca.crt'),
        "key": os.path.join(res, 'certs', 'server.key'),
        "cert": os.path.join(res, 'certs', 'server.crt'),
        "dh": os.path.join(res, 'certs', 'dh.pem'),
        "users": os.path.join(res, 'scripts', 'users.txt'),
        "auth_sh": os.path.join(res, 'scripts', 'check_auth.sh')
    }

def get_rotation_date():
    addon = xbmcaddon.Addon()
    date = addon.getSetting('last_rotation')
    return date if date else "Never"

def show_status():
    date = get_rotation_date()
    message = (
        f"Last Key Rotation: [COLOR gold]{date}[/COLOR]\n"
        "Encryption: RSA-2048 / AES-256-GCM\n"
        "Status: All systems secure."
    )
    xbmcgui.Dialog().ok("VPN Security Status", message)

def generate_certs(is_rotation=False):
    addon = xbmcaddon.Addon()
    p = get_paths()
    certs_dir = os.path.join(p["res"], "certs")
    cnf_path = p["cnf"]
    
    if not os.path.exists(cnf_path):
        xbmcgui.Dialog().ok("Error", f"Missing critical file: {cnf_path}")
        return

    if is_rotation:
        if os.path.exists(certs_dir):
            shutil.rmtree(certs_dir)
            xbmc.log("VPN_PORTABLE: Certificates wiped for rotation.", xbmc.LOGINFO)
    
    if not os.path.exists(certs_dir):
        os.makedirs(certs_dir)

    conf = f"OPENSSL_CONF={cnf_path}"

    try:
        dialog = xbmcgui.DialogProgress()
        title = "Security Rotation" if is_rotation else "VPN Setup"
        dialog.create(title, "Generating Massive Encryption...")

        dialog.update(20, "Creating New Master CA...")
        os.system(f"{conf} openssl req -new -x509 -extensions v3_ca -nodes -days 3650 -keyout '{os.path.join(certs_dir, 'ca.key')}' -out '{p['ca']}' -subj '/CN=OpenVPN-CA'")

        dialog.update(40, "Creating New Server Identity...")
        os.system(f"{conf} openssl req -new -nodes -keyout '{p['key']}' -out '{os.path.join(certs_dir, 'server.csr')}' -subj '/CN=OpenVPN-Server'")

        dialog.update(60, "Signing Server Authority...")
        os.system(f"{conf} openssl x509 -req -extensions server_cert -extfile '{cnf_path}' -in '{os.path.join(certs_dir, 'server.csr')}' -CA '{p['ca']}' -CAkey '{os.path.join(certs_dir, 'ca.key')}' -CAcreateserial -out '{p['cert']}' -days 3650")

        dialog.update(80, "Generating DH (This is the heavy math)...")
        os.system(f"openssl dhparam -out '{p['dh']}' 2048")

        dialog.close()
        
        now = datetime.now().strftime("%Y-%m-%d %H:%M")
        addon.setSetting('last_rotation', now)
        xbmc.executebuiltin('Container.Refresh')
        
        msg = "Keys Rotated! VPN will now restart." if is_rotation else "Addon setup complete."
        if xbmcgui.Dialog().ok("Success", msg):
            stop_server()
            start_server()

    except Exception as e:
        if 'dialog' in locals(): dialog.close()
        xbmcgui.Dialog().ok("Python Error", str(e))

def update_credentials():
    addon = xbmcaddon.Addon()
    p = get_paths()
    user = addon.getSetting('vpn_user').strip()
    pwd = addon.getSetting('vpn_pwd').strip()
    cred_path = p["users"]

    try:
        with open(cred_path, 'w') as f:
            f.write(f"{user}\n{pwd}")
        os.chmod(cred_path, 0o600)
        xbmc.log(f"VPN_PORTABLE: Updated {cred_path}", xbmc.LOGINFO)
        return True
    except Exception as e:
        xbmc.log(f"VPN_PORTABLE ERROR: {str(e)}", xbmc.LOGERROR)
        return False

def start_server():
    update_credentials() 
    p = get_paths()
    os.system("killall -9 openvpn 2>/dev/null")
    xbmc.sleep(500)
    
    os.system("sysctl -w net.ipv4.ip_forward=1")
    os.system("iptables -t nat -A POSTROUTING -s 10.8.0.0/24 -o eth0 -j MASQUERADE 2>/dev/null || iptables -t nat -A POSTROUTING -s 10.8.0.0/24 -o wlan0 -j MASQUERADE")
  
    cmd = [p["bin"], "--config", "server.conf", "--log-append", "logs/openvpn.log"]
    try:
        subprocess.Popen(cmd, cwd=p["res"])
        xbmcgui.Dialog().notification("VPN", "Server Started", "info", 3000)
    except Exception as e:
        xbmcgui.Dialog().ok("Start Error", str(e))

def stop_server():
    os.system("killall -9 openvpn 2>/dev/null")
    xbmcgui.Dialog().notification("VPN", "Server Stopped", "info", 3000)

def view_server_log():
    p = get_paths()
    if os.path.exists(p["log"]):
        with open(p["log"], 'r') as f:
            content = "".join(f.readlines()[-50:])
        xbmcgui.Dialog().textviewer("VPN Log", content)
    else:
        xbmcgui.Dialog().ok("Log Missing", "No log found.")

def export_client_ovpn():
    addon = xbmcaddon.Addon()
    p = get_paths()
    server_address = addon.getSetting('server_address') or "yourname.duckdns.org"
    filename = addon.getSetting("ovpn_filename").strip().replace(" ", "_") or "client_phone"
    if not filename.endswith(".ovpn"): filename += ".ovpn"

    try:
        if not os.path.exists(p["ca"]):
            raise Exception("CA certificate not found!")
        with open(p["ca"], "r") as f:
            ca_content = f.read().strip()

        ovpn_content = f"client\ndev tun\nproto udp\nremote {server_address} 1194\nresolv-retry infinite\nnobind\npersist-key\npersist-tun\nauth-nocache\ncipher AES-256-GCM\nauth SHA256\ntls-version-min 1.2\nmssfix 1400\nremote-cert-tls server\nauth-user-pass\nsetenv CLIENT_CERT 0\n<ca>\n{ca_content}\n</ca>\nverb 3\n"

        # 3. Save the file
        export_folder = xbmcgui.Dialog().browse(3, "Select Export Location", "files")
        
        # If user didn't pick a folder, use the Addon's script folder as a fallback
        if not export_folder:
            export_folder = os.path.join(p["res"], "scripts")
            xbmcgui.Dialog().notification("VPN", "No folder picked, using default scripts folder", "info", 3000)
        
        full_dest = os.path.join(export_folder, filename)
        
        with open(full_dest, "w") as f:
            f.write(ovpn_content)
            
        xbmcgui.Dialog().ok("Export Successful", f"File saved to:\n{full_dest}")
    except Exception as e:
        xbmcgui.Dialog().ok("Export Error", str(e))

def clear_vpn_log():
    p = get_paths()
    if xbmcgui.Dialog().yesno("Clear Log", "Wipe log history?"):
        with open(p["log"], 'w') as f: f.write("--- Log Cleared ---\n")

# --- MAIN EXECUTION ---

if __name__ == '__main__':
    # Parse Arguments
    params = dict(arg.split('=') for arg in sys.argv[1:] if '=' in arg)
    action = params.get('action', None)

    if action == "clear_log": clear_vpn_log()
    elif action == "generate_certs": generate_certs(False)
    elif action == "rotate_certs":
        if xbmcgui.Dialog().yesno("RESET?", "Rotate Keys?"): generate_certs(True)
    elif action == "export_phone": export_client_ovpn()
    elif action == "start": start_server()
    elif action == "stop": stop_server()
    elif action == "view_log": view_server_log()
    elif action == "show_status": show_status()

    # DASHBOARD (When clicking the Addon Icon)
    else:
        is_running = os.system("/usr/bin/pgrep -x openvpn > /dev/null") == 0
        status_text = "[COLOR green]● ONLINE[/COLOR]" if is_running else "[COLOR red]○ OFFLINE[/COLOR]"
        options = ["Start Server", "Stop Server", "View Log", "Clear Log", "Export for Phone", "Status Info", "Rotate Keys", "Open Settings"]
        sel = xbmcgui.Dialog().select(f"VPN Dashboard: {status_text}", options)
        
        if sel == 0: start_server()
        elif sel == 1: stop_server()
        elif sel == 2: view_server_log()
        elif sel == 3: clear_vpn_log()
        elif sel == 4: export_client_ovpn()
        elif sel == 5: show_status()
        elif sel == 6: generate_certs(True)
        elif sel == 7: xbmcaddon.Addon().openSettings()