#!/bin/bash

IMG=$(echo *.img.gz)
if [ -z "$IMG" ]; then
    echo 'No .img.gz found'
    exit 1
fi

# Be root to run this script
if [ $UID -ne 0 ]; then
  echo 'Please, type root password...'
  su -c "$0 $@"
  exit
fi

# Get the device
DEVICE=''
if [ "$DEVICE" == "" ]; then
  DEVICES=$(ls /sys/block | egrep 'sd[a-z]$')
  for D in $DEVICES; do
    DETACHABLE=$(cat "/sys/block/$D/removable")
    if [ "$DETACHABLE" -eq 1 ]; then
      if [ -e "/sys/block/$D/$D1" ]; then
        echo
        read -e -p 'Confirm USB device: ' -i "/dev/$D" DEVICE
        break
      fi
    fi
  done
fi

if [ "$DEVICE" == "" ]; then
  echo 'No device given.'
  exit 2
elif [ ! -e "$DEVICE" ]; then
  echo "$DEVICE does not exist."
  exit 2
fi

# Check the device if it's a pen drive
DEVICEBN=$(basename $DEVICE)
DETACHABLE=$(cat "/sys/block/$DEVICEBN/removable")
if [ "$DETACHABLE" -eq 1 ]; then
  if [ ! -e "/sys/block/$DEVICEBN/$DEVICEBN1" ]; then
    echo "$DEVICE has no partition."
    exit 3
  fi
else
  echo "$DEVICE is not a detachable device."
  exit 3
fi
  
# Make sure the device is not in use and not mounted
SDDIR=$(grep $DEVICE'1' /etc/mtab | awk '{print $2}' | sed 's/\\040/ /g')
if [ "$SDDIR" == "" ]; then
  SDDIR=$(grep $DEVICE'2' /etc/mtab | awk '{print $2}' | sed 's/\\040/ /g')
fi
if [ "$SDDIR" != "" ]; then
  # Is it in use?
  FUSER=$(fuser -m $SDDIR)
  if [ "$FUSER" != "" ]; then
    echo "$DEVICE is in use. Close any programs using the device."
    exit 5
  else
    # Make sure the USB is not mounted
    udisksctl unmount -b $DEVICE'1'
    udisksctl unmount -b $DEVICE'2'
  fi
fi

# Write image to SD
echo "Clean $DEVICE..."
dd if=/dev/zero of=$DEVICE bs=1M count=1
echo "Write $IMG to $DEVICE..."
gunzip -k -c $IMG | dd of=$DEVICE bs=64k oflag=dsync status=progress
echo "Grow ${DEVICE}2..."
resize2fs -p ${DEVICE}2


# Prepare SD for development
DTB=$(echo *.dtb)
if [ ! -z "$DTB" ]; then
    SDDIR='/mnt/sd'
    mkdir $SDDIR 2>/dev/null
    mount $DEVICE'2' $SDDIR
    cp -v "$DTB" "$SDDIR"
    sed -i "s#/.*\.dtb#/$DTB#" "$SDDIR/extlinux/extlinux.conf"
    umount $SDDIR
    rmdir $SDDIR 2>/dev/null
fi

echo 'Done!'
