# -*- coding: utf-8 -*-

'''
    Kodious
    Copyright (C) 2016 Hamid_PaK

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
'''

import os
import sys
import cPickle
import hashlib

from datetime import datetime, timedelta


class Cachable(object):
    def __init__(self, path, prefix = 'cache_file_'):
        # creates a unique file name
        filepath = os.path.join(path, prefix)
        sha1 = hashlib.sha1(filepath).hexdigest()

        # sets the filepath
        self.filepath = filepath + sha1

        if (not self.__accessPath()) or (not self.__accessFile()):
            self.__create()


    def get(self, key, override = False):
        """Get data from the cache.

        All data stored under *key* is returned. If the data is expired,
        None is returned. Expired data is returned if *override* is True.

        Args:
            key: (string) the name of the data to fetch.
            override: (bool) return expired data; defaults to False.

        Returns:
            the requested data or None if the requested data has expired or not existed.

        """
        try:
            data = self.__read()
        except:
            data = {}

        if not isinstance(key, str):
            key = self.__generateKey(key)

        try:
            value = data[key]['data']
            expires = data[key]['expires']
        except KeyError:
            return

        if (expires and expires < datetime.now() and not override):
            return

        return value


    def set(self, key, value, timeout = None):
        """Store data in the cache.

        The data, *value*, is stored under the name, *key*, in the cache.
        The data must be picklable. Optionally, the data can expire after
        *timeout* seconds have passed.

        Args:
            key: (string) the name given to the data.
            value: the data to be stored.
            timeout: (int, long, float, or None) how long in seconds the data
                should be considered valid -- if None, defaults to forever.

        """
        try:
            data = self.__read()
        except:
            data = {}

        if timeout is None:
            expires = None
        else:
            expires = (datetime.now() + timedelta(seconds=timeout))

        if not isinstance(key, str):
            key = self.__generateKey(key)

        data[key] = {'expires': expires, 'data': value}
        self.__write(data)

        return value


    def flush(self):
        self.__write({})


    def __accessFile(self):
        """Checks the accessibility of cache path"""
        return os.path.isfile(self.filepath)


    def __accessPath(self):
        """Checks for file access"""
        return os.access(os.path.dirname( self.filepath ), os.F_OK)


    def __create(self):
        """Create a cache file."""
        if not self.__accessPath():
            os.makedirs(os.path.dirname( self.filepath ))

        if not self.__accessFile():
            self.__write({})


    def __read(self):
        """Open a file and use cPickle to retrieve its data"""
        with open(self.filepath, 'rb') as f:
            return cPickle.load(f)


    def __write(self, data):
        """Write to cache file"""
        with open(self.filepath, 'w+b') as f:
            cPickle.dump(data, f, cPickle.HIGHEST_PROTOCOL)


    def __generateKey(self, data):
        return hashlib.sha1(str(data)).hexdigest()