# -*- coding: utf-8 -*-

'''
    AccuRadio Audio Addon
    Copyright (C) 2016 Hamid_PaK

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
'''

import re
import random
import urllib
import urlparse

from urlparse import urljoin
from resources.lib.kodious.module import AddonModule


class accuRadioBase(AddonModule):

    # Complete URLs
    ROOT_URL = 'https://www.accuradio.com'

    # JSON URLs
    JSON_BASE_URL = urljoin(ROOT_URL, '/c/m/json/')

    # Listener URLs
    LOGIN_URL = urljoin(ROOT_URL, '/listener/dj-login/2013/')
    LOGOUT_URL = urljoin(ROOT_URL, '/listener/all-logout/')
    # CSRF token URL
    CSRF_TOKEN_URL = urljoin(ROOT_URL, '/listener/dj-csrf/')

    # User URLS
    USER_STAUS_URL = urljoin(ROOT_URL, '/listener/user-status/')

    USER_FAVOURITES_URL = urljoin(ROOT_URL, JSON_BASE_URL + 'favorites/')
    USER_FAVOURITES_ADD_URL = urljoin(ROOT_URL, USER_FAVOURITES_URL + 'add/')
    USER_FAVOURITES_REMOVE_URL = urljoin(ROOT_URL, USER_FAVOURITES_URL + 'remove/')

    # Channel URLS
    CHANNEL_BROWSER_URL = urljoin(ROOT_URL, JSON_BASE_URL + 'browser/')
    CHANNEL_BROWSER_ID_URL_FMT = urljoin(ROOT_URL, CHANNEL_BROWSER_URL + '%s/')
    CHANNEL_BROWSER_RANDOM_URL = urljoin(ROOT_URL, JSON_BASE_URL + 'random_channels/')
    CHANNEL_IMAGE_URL_FMT = 'http://c2.accu.fm/tiles/default/%s.jpg'

    CHANNEL_URL = urljoin(ROOT_URL, JSON_BASE_URL + 'channel/')
    TRACK_HISTORY_URL = urljoin(ROOT_URL, '/listener/track-history/')

    # Partial URL
    PLAYLIST_URL_FMT = urljoin(ROOT_URL, 'playlist/json/%s/')
    ALMBUM_COVER_URL_FMT = 'http://cdn.accuradio.com/static/images/covers300/%s'


    def __init__(self):
        AddonModule.__init__(self)

        self._user_id = None
        self._keep_user_logged_in = True
        self._csrf_token = None
        self._csrf_token = self.getCSFRToken()

        creds = self.userGetCredentials()

        if creds and not self.userIsLoggedIn():
            self.userLogin( creds[0], creds[1] )

        self._cache_timeout = int(self.getSetting('cache.expire') or 7) * 86400
        if not self._cache_timeout:
            self._cache_timeout = None


    def _get_headers(self, isAJAX=False):
        headers = {
            'Accept': '*/*',
            'Host': 'www.accuradio.com',
            'Referer': self.ROOT_URL,
            'User-Agent': 'AccuRadio/3.0 CFNetwork/700.0.0 Android/2.0.0',
            'Accept-Language': 'en-us',
            'Accept-Encoding': 'gzip, deflate',
        }

        if self._csrf_token:
            headers['Cookie'] = 'csrftoken=%s' % (self._csrf_token)
            headers['X-CSRFToken'] = self._csrf_token
            headers['HTTP_X_CSRFTOKEN'] = self._csrf_token

        if isAJAX:
            headers[ 'X-Requested-With' ] = 'XMLHttpRequest'

        return headers


    def httpRequest(self, url, postData=None, isAJAX=False):
        return self.http.request( url, headers=self._get_headers(isAJAX), postData=postData )

    def httpGet(self, url, query=None, isAJAX=False):
        if query:
            if isinstance(query, dict):
                url += '?' + urllib.urlencode(query)
            elif isinstance(query, basestring):
                url += '?' + query

        return self.httpRequest(url, isAJAX=isAJAX)


    def getCSFRToken(self, forceRefresh=False):
        cache_key = 'accuRadioBase.csrf_token'
        token = self.cache.get( cache_key )

        if not token or forceRefresh:
            with self.httpRequest(self.CSRF_TOKEN_URL, isAJAX=False) as res:
                if res.isOkay():
                    token = res.html()
                    self.cache.set( cache_key, token, 3600 )

        assert token != None

        return token


    def userLogin(self, email, password):
        data = {
            'email': email,
            'password': password,
            'site': 'mobile',
        }
        with self.httpRequest(self.LOGIN_URL, postData=data) as res:
            if res.isOkay():
                result = res.json()

                if result and result.get('status') == 'ok':
                    username = result.get('username')

                    if username:
                        self.userIsLoggedIn( username )
                        self.getCSFRToken( True )

                    return True

        return False


    def userLogout(self):
        with self.httpGet( self.LOGOUT_URL, query='callback=jQuery' ) as res:
            print res.html()
            if res.isOkay():
                self._user_id = None
                cache_key = 'accuRadioBase.user_id'
                self.cache.set( cache_key, None )


    def userIsLoggedIn(self, user_id=None):
        cache_key = 'accuRadioBase.user_id'

        if user_id:
            # Stores the user information for one hour
            self.cache.set( cache_key, user_id, 3600 )

        else:
            user_id = self.cache.get( cache_key )

        if user_id:
            # Sets the user ID for instance
            self._user_id = user_id

        return user_id or False


    def userGetCredentials(self):
        email = self.getSetting('login.email')
        encrypted_pass = self.getSetting('login.password')
        decrypted_pass = None

        if email and encrypted_pass:

            if not encrypted_pass.startswith('{{') and not encrypted_pass.endswith('}}'):
                decrypted_pass = encrypted_pass

                encrypted_pass = '{{%s}}' % self.encryptB64(decrypted_pass, email)
                self.setSetting('login.password', encrypted_pass)

            if not decrypted_pass:
                try: encrypted_pass = re.compile('^\{\{(.*)\}\}$').search( encrypted_pass ).group(1)
                except: return False
                decrypted_pass = self.decryptB64(encrypted_pass, email)

            return (email, decrypted_pass)
        else:
            return False


    def userGetFavouriteChannels(self):
        q = {
            'u': self._user_id,
        }

        with self.httpGet( self.USER_FAVOURITES_URL, query=q ) as res:
            if res.isOkay():
                result = res.json()

                if result and result.get('status') == 'ok':

                    items = []
                    for item in result.get('channels'):
                        items.append({
                            'id': item['_id']['$oid'],
                            'title': item['name'],
                            'cover': self.CHANNEL_IMAGE_URL_FMT % item.get('uoldid'),
                            'desc': item.get('description'),
                        })

                    return items


    def userAddFavouriteChannel(self, channelID, channelTitle):
        assert isinstance(channelID, basestring) and isinstance(channelTitle, basestring)

        data = {
            'name': channelTitle,
            'o': channelID,
            'u': self._user_id,
        }

        oid = None
        with self.httpRequest( self.CHANNEL_URL, postData=data ) as res:
            if res.isOkay():
                result = res.json()

                oid = result['_id']['$oid']

        assert oid != None

        data = {
            'c': oid,
            'u': self._user_id,
        }

        with self.httpRequest( self.USER_FAVOURITES_ADD_URL, postData=data ) as res:
            if res.isOkay():
                result = res.json()

                return result and result.get('status') == 'ok'

        return False


    def removeUserFavouriteChannel(self, channelID):
        assert isinstance(channelID, basestring)

        data = {
            'c': channelID,
            'u': self._user_id,
        }

        with self.httpRequest( self.USER_FAVOURITES_REMOVE_URL, postData=data ) as res:
            if res.isOkay():
                result = res.json()

                return result and result.get('status') == 'ok'

        return False


    def trackHistory(self, trackID=None, channelID=None):
        if trackID and channelID:

            data = {
                't': trackID,
                'c': channelID,
            }

            if self._user_id:
                data[ 'u' ] = self._user_id

        else:
            data = None

        with self.httpRequest( self.TRACK_HISTORY_URL, postData=data ) as res:
            if res.isOkay():
                result = res.json()

                if result:
                    if isinstance(result, dict) and result.get('status'):
                        return True
                    elif isinstance(result, list):
                        items = {}
                        for x in result:
                            try:
                                oid = x['$oid']
                                items[ oid ] = True
                            except KeyError: pass

                        return items

        return False


    def trackChannel(self, channelID=None):
        if channelID:

            token = self.getCSFRToken()

            data = {
                'o': channelID,
                'getts': 1,
                'getando': 1,
                'csrfmiddlewaretoken': token,
            }

        else:
            data = None

        with self.httpRequest( self.CHANNEL_URL, postData=data ) as res:
            if res.isOkay():

                result = res.json()

                return result

        return False


    def getGenres(self, noCache=False, timeCache=None):
        cache_key = 'accuRadioBase.main.genres'

        items = self.cache.get( cache_key )

        if (not noCache) and items: return items

        with self.httpGet( self.CHANNEL_BROWSER_URL ) as res:
            if res.isOkay():

                result = res.json()

                if result and result.get('status') == 'ok' and result.get('brands'):
                    items = []
                    for item in result.get('brands'):
                        items.append({
                            'id': item[ 'param' ],
                            'title': item[ 'name' ],
                        })

                    # stores the genres in cache
                    self.cache.set( cache_key, items, timeCache or self._cache_timeout )

                    return items
        pass


    def getChannels(self, genreID=None, noCache=False, timeCache=None):
        if not genreID:
            genreID = 'random'
            url = self.CHANNEL_BROWSER_RANDOM_URL
        else:
            url = self.CHANNEL_BROWSER_ID_URL_FMT % genreID

        cache_key = 'accuRadioBase.main.channels.genre_id#%s' % genreID

        items = self.cache.get(cache_key)

        if (not noCache) and items: return items

        with self.httpGet( url ) as res:
            if res.isOkay():
                result = res.json()

                if result and result.get('status') == 'ok' and result.get('channels'):
                    items = []
                    for item in result.get('channels'):
                        items.append({
                            'id': item['_id']['$oid'],
                            'title': item['name'],
                            'cover': self.CHANNEL_IMAGE_URL_FMT % item.get('oldid'),
                            'desc': item.get('description'),
                        })

                    # stores the channels of the selected genre in cache
                    self.cache.set( cache_key, items, timeCache or self._cache_timeout )

                    return items


    def getChannelPlaylist(self, channelID, noCache=True, timeCache=0):
        cache_key = 'accuRadioBase.main.channels.playlistchannel_id#%s' % channelID

        items = self.cache.get( cache_key )

        if (not noCache) and items: return items

        url = self.PLAYLIST_URL_FMT % channelID

        with self.httpGet( url ) as res:
            if res.isOkay():
                json_list = res.json()

                assert json_list

                items = {}
                for x in json_list:
                    try:
                        oid = x['id']['$oid']
                        album = x['album']
                        item = {
                            'oid': oid,
                            'cid': channelID,
                            'title': x['title'],
                            'duration': x['duration'],
                            'artist': x['track_artist'],
                            'artistdisplay': x['artist']['artistdisplay'],
                            'urls': [
                                x['primary'] + x['fn'] + '.m4a',
                                x['secondary'] + x['fn'] + '.m4a',
                            ],
                            'album': {
                                'title': album['title'],
                                'label': album['label'],
                                'year': album['year'],
                                'cdcover': self.ALMBUM_COVER_URL_FMT % album['cdcover'].strip('/'),
                            },
                        }

                        items[ oid ] = item
                    except KeyError: pass

                # Note: We can create a DB of songs
                # self.cache.set( cache_key, items )

            return items
        pass
