# -*- coding: utf-8 -*-

'''
    Kodious
    Copyright (C) 2016 Hamid_PaK

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
'''

from ..kodi import *
from ..routes import RouteAction


class BaseListItem(object):
    def __init__(self,
        path=None,
        label=None,
        infoType=None,
        infoValues=None,
        artValues=None,
        isFolder=False,
    ):
        '''
        Returns a new instance of XBMC/Kodi ListItem from xbmcgui

        Keyword arguments:
            path: string -- URL of list item (default None)
            label: string -- Title of list item (default None)
            infoType: string -- Extra option if infoList is being provided (default None)
        '''

        # available infos
        self._infoList = [
            'count', 'size', 'date'
        ]

        self._artList = [
            'thumb', 'poster', 'banner', 'fanart', 'clearart', 'clearlogo', 'landscape', 'icon'
        ]

        self._path = path
        self._label = label
        self._infoType = infoType or ''
        self._infoValues = infoValues or {}
        self._artValues = artValues or {}
        self._isFolder = isFolder
        self._properties = {}

        return self


    def setLabel(self, label):
        '''
        Sets the listitem's label.

        Parameters
            label   string or unicode - text string.
        Example:
            listitem.setLabel('Casino Royale')
        '''
        self._label = label


    def getLabel(self):
        return self._label


    def setLabel2(self, label2):
        '''
        Sets the listitem's label2.

        Parameters
            label   string or unicode - text string.
        Example:
            listitem.setLabel2('Casino Royale')
        '''
        self._label2 = label2


    def getLabel2(self):
        return self._label2


    def setInfo(self, values):
        '''
        Sets the listitem's infoLabels.

        Parameters
            infoLabels  dictionary - pairs of { label: value }
        '''
        if isinstance(values, dict):
            self._infoValues.update( values )


    def getInfo(self):
        info = {}
        for x in self._infoList:
          if (x in self._infoValues.keys()):
            info[ x ] = self._infoValues[ x ]

        return info


    def setArt(self, values):
        '''
        Sets the listitem's art

        Parameters:
            values  dictionary - pairs of { label: value }.
            Some default art values (any string possible):
        Label:
            thumb   string - image filename
            poster  string - image filename
            banner  string - image filename
            fanart  string - image filename
            clearart    string - image filename
            clearlogo   string - image filename
            landscape   string - image filename
            icon    string - image filename
        '''
        if isinstance(values, dict):
            self._artValues.update( values )


    def getArt(self):
        infos = {}
        for x in self._artList:
          if (x in self._artValues.keys()):
            infos[ x ] = self._artValues[ x ]

        return infos


    def setCast(self, actors):
        '''
        Set cast including thumbnails

        Parameters:
            actors  list of dictionaries (see below for relevant keys)
        Keys:
            name    string (Michael C. Hall)
            role    string (Dexter)
            thumbnail   string (http://www.someurl.com/someimage.png)
            order   integer (1)
        Example:
            actors = [{"name": "Actor 1", "role": "role 1"}, {"name": "Actor 2", "role": "role 2"}]
            listitem.setCast(actors)
        '''
        if isinstance(actors, dict):
            self._cast = actors


    def setPath(self, path):
        '''
        Sets the listitem's path.

        Parameters:
            path    string or unicode - path, activated when item is clicked.
        Note:
            You can use the above as keywords for arguments.
        '''
        self._path = unicode( path )


    def getPath(self):
        return self._path


    def setProperty(self, key, value):
        self._properties.update({ key: value })


    def getProperty(self, key):
        return self._properties[ key ]


    def setUniqueIDs(self, values):
        '''
        Sets the listitem's uniqueID

        Parameters
            values  dictionary - pairs of { label: value }.
        Some example values (any string possible):
            imdb    string - uniqueid name
            tvdb    string - uniqueid name
            tmdb    string - uniqueid name
            anidb   string - uniqueid name
        '''
        if isinstance(values, dict):
            self._uniqueIDs = values


    def setSubtitles(self, files):
        '''
        Sets subtitles for this listitem.

        Parameters:
            subtitleFiles   list with path to subtitle files
        Example:
            listitem.setSubtitles(['special://temp/example.srt', 'http://example.com/example.srt'])
        '''
        if isinstance(files, list):
            self._subtitles = files


    def setRating(self, _type, rating, votes=0, defaultt=False):
        '''
        Sets a listitem's rating. It needs at least type and rating param

        Parameters:
            type    string - the type of the rating. Any string.
            rating  float - the value of the rating.
            votes   int - the number of votes. Default 0.
            defaultt    bool - is the default rating?. Default False.
        Some example type (any string possible):
            imdb    string - rating type
            tvdb    string - rating type
            tmdb    string - rating type
            anidb   string - rating type
        Example:
            listitem.setRating("imdb", 4.6, 8940, True)
        '''
        rateTypes = [ 'imdb', 'tvdb', 'tmdb', 'anidb' ]

        if _type in rateTypes:
            self._rating = {
                'type': _type,
                'rating': rating,
                'votes': votes,
                'defaultt': defaultt,
            }


    def getRating(self, key):
        try: return self._rating.get(key)
        except: return {}


    def setMimeType(self, mimetype):
        '''
        Sets the listitem's mimetype if known.

        Parameters:
            mimetype    string or unicode - mimetype
        If known prehand, this can (but does not have to) avoid HEAD requests being sent to HTTP servers to figure out file type.
        '''
        self._mimetype = mimetype


    def setSelected(self, selected):
        '''
        Sets the listitem's selected status.

        Parameters:
            selected    bool - True=selected/False=not selected
        Example:
            listitem.select(True)
        '''
        self._selected = bool( selected )


    def isSelected(self):
        return self._selected


    def setPlayable(self, value):
        str_bool = ['false', 'true']
        self.setProperty('IsPlayable', str_bool[ bool( value )])


    def isPlayable(self):
        self.getProperty('IsPlayable')


    def addStreamInfo(self, _type, values):
        '''
        Add a stream with details.

        Parameters:
            type    string - type of stream(video/audio/subtitle).
            values  dictionary - pairs of { label: value }.

            Video Values:
                codec   string (h264)
                aspect  float (1.78)
                width   integer (1280)
                height  integer (720)
                duration    integer (seconds)
            Audio Values:
                codec   string (dts)
                language    string (en)
                channels    integer (2)
            Subtitle Values:
                Label   Description
                language    string (en)
        Example:
            listitem.addStreamInfo('video', { 'codec': 'h264', 'width' : 1280 })
        '''
        self._streamInfo = {
            'type': _type,
            'values': values,
        }


    def addContextMenuItems(self, items, replaceItems=False):
        '''
        Adds item(s) to the context menu for media lists.

        Parameters:
            items   list - [(label, action,)*] A list of tuples consisting of label and action pairs.
                label string or unicode - item's label.
                action string or unicode - any built-in function to perform.
            replaceItems    boolean - Default is False
        Note:
            You can use the above as keywords for arguments and skip certain optional arguments.
            Once you use a keyword, all following arguments require the keyword.
        Example:
            listitem.addContextMenuItems([('Theater Showtimes', 'RunScript(special://home/scripts/showtimes/default.py,Iron Man)',)])
        '''
        if isinstance(items, list):
            self._contextMenuItems = {
                'items': items,
                'replaceItems': replaceItems,
            }


    def getListItem(self):
        listitem = xbmcgui.ListItem()

        # Sets all the properties
        for key, value in self._properties.iteritems():
            try:
                listitem.setProperty( key, value )
            except: pass

        # Sets the path
        try: listitem.setPath(self._path)
        except: pass

        # Sets the label
        try: listitem.setLabel(self._label)
        except: pass

        # Sets the label2
        try: listitem.setLabel2(self._label2)
        except: pass

        # Sets the info labels
        try: listitem.setInfo(self._infoType, self.getInfo())
        except: pass

        # Sets the arts
        # Older version of XBMC
        try: listitem.setIconImage(self.getArt()['icon'])
        except: pass
        try: listitem.setThumbnailImage(self.getArt()['thumb'])
        except: pass
        # New version of XBMC
        try: listitem.setArt(self.getArt())
        except: pass

        # Set the unique IDs
        try: listitem.setUniqueIDs(self._uniqueIDs)
        except: pass

        # Sets the subtitles
        try: listitem.setSubtitles(self._subtitles)
        except: pass

        # Sets the rating
        try: listitem.setRating(**self._rating)
        except: pass

        # Sets the mime type
        try: listitem.setMimeType(self._mimetype)
        except: pass

        # Sets selected
        try: listitem.select(self._selected)
        except: pass

        # Sets the menu items
        try: listitem.addContextMenuItems(**self._contextMenuItems)
        except: pass

        # Sets the stream info
        try: listitem.addStreamInfo(**self._streamInfo)
        except: pass

        return listitem


    def asTuple(self):
        return (self._path, self.getListItem(), self._isFolder)
