#!/bin/bash


BLUE='\033[1;34m'
NC='\033[0m' # No Color

set -e

HTTP_DIR='http://jernej.libreelec.tv/updates/H3/'
BASE_DIR='/storage/downloads'
TAR_ROOT_DIR='target'
TEMP_DIR='updates_tmp'
FLASH_DIR='/flash'
FLASH_TEMP_DIR='.target_tmp'
BACKUP_DIR='.backup'
RELEASE_REMOTE_PREFIX='LibreELEC-H3.arm-[0-9]\+.[0-9]\+.\?[0-9]*-devel-'
RELEASE_LOCAL_PREFIX='devel-'


RUNNING_RELEASE=$(lsb_release -b)
RUNNING_RELASE_DT=$(echo "$RUNNING_RELEASE" | grep -o $RELEASE_LOCAL_PREFIX'[0-9]\+-' | grep -o '[0-9]*')

echo -e ${BLUE}"Current release is $RUNNING_RELEASE"${NC}

if [ "$1" != "" ]; then
	LATEST_RELEASE_URL="$1"
	LATEST_RELEASE=$(basename -- "$LATEST_RELEASE_URL")
	FORCE_UPDATE="1"
else
	REMOTE_RELEASES=$(curl -s $HTTP_DIR | grep -o 'href="'$RELEASE_REMOTE_PREFIX'[0-9]\+-[0-9a-zA-Z]\+\.tar"' | grep -o $RELEASE_REMOTE_PREFIX'.*\.tar' | sort -fbi)
	LATEST_RELEASE=$(echo "$REMOTE_RELEASES" | tail -n 1)
	LATEST_RELEASE_URL=$HTTP_DIR$LATEST_RELEASE
	FORCE_UPDATE="0"
fi
LATEST_RELEASE_DT=$(echo "$LATEST_RELEASE" | grep -o $RELEASE_LOCAL_PREFIX'[0-9]\+-' | grep -o '[0-9]*')

if [ "$LATEST_RELEASE_DT" -gt "$RUNNING_RELASE_DT" ] || [ "$FORCE_UPDATE" == "1" ]; then
	if [ "$FORCE_UPDATE" == "1" ]; then
		echo -e ${BLUE}"Using release from $LATEST_RELEASE_URL"${NC}
	else
		echo -e ${BLUE}"Found new release $LATEST_RELEASE"${NC}
	fi
	while true; do
		echo -e -n ${BLUE}"Proceed downloading and applying update? [Y]es, [N]o or [C]ancel: "
		read confirmation
		if [ ${confirmation:0:1} == "Y" ]; then
			echo -e -n ${NC}
			break
		elif [ ${confirmation:0:1} == "N" ] || [ ${confirmation:0:1} == "C" ]; then
			echo -e "Update aborted."${NC}
			exit 1
		else
			echo -e "Invalid input \"$confirmation\""${NC}
		fi
	done
else
    echo -e ${BLUE}"Latest release is $LATEST_RELEASE"${NC}
	echo -e ${BLUE}"System is up to date! Nothing to do."${NC}
	exit 0
fi

echo -e ${BLUE}"Creating temporary download directory"${NC}
cd $BASE_DIR/
mkdir $TEMP_DIR
cd $TEMP_DIR/
echo -e ${BLUE}"Downloading release from $LATEST_RELEASE_URL"${NC}
wget $LATEST_RELEASE_URL
wget $LATEST_RELEASE_URL'.sha256'
echo -e ${BLUE}"Checking download integrity"${NC}
sha256sum -c $LATEST_RELEASE'.sha256'
echo -e ${BLUE}"Unpacking"${NC}
tar -xf $LATEST_RELEASE
cd "${LATEST_RELEASE%.*}"/
echo -e ${BLUE}"Checking images integrity"${NC}
md5sum -c "$TAR_ROOT_DIR"/*.md5
echo -e ${BLUE}"Remounting flash storage as R/W"${NC}
mount -o remount,rw $FLASH_DIR

FREE_SPACE_KB=$(df -k $FLASH_DIR | tail -1 | awk '{print $4}')
if [ -d "$FLASH_DIR/$BACKUP_DIR" ]; then
	BACKUP_SIZE=$(du -k $FLASH_DIR/$BACKUP_DIR | awk '{print $1}')
else
	BACKUP_SIZE="0"
fi
REQUIRED_SIZE=$(du -k "$TAR_ROOT_DIR" | awk '{print $1}')
if [ $((FREE_SPACE_KB + BACKUP_SIZE)) -lt "$REQUIRED_SIZE" ]; then
	echo -e ${BLUE}"Not enough space to create backup files."${NC}
	while true; do
		echo -e -n ${BLUE}"Proceed overwriting images? [Y]es, [N]o or [C]ancel: "
		read confirmation
		if [ ${confirmation:0:1} == "Y" ]; then
			echo -e "Copying files"${NC}
			cp -af "$TAR_ROOT_DIR"/* $FLASH_DIR/
			break
		elif [ ${confirmation:0:1} == "N" ] || [ ${confirmation:0:1} == "C" ]; then
			echo -e "Update aborted."${NC}
			exit 1
		else
			echo -e "Invalid input \"$confirmation\""${NC}
		fi
	done
else
	rm -rf "$FLASH_DIR/$BACKUP_DIR"
	rm -rf "$FLASH_DIR/$FLASH_TEMP_DIR"
	mkdir "$FLASH_DIR/$FLASH_TEMP_DIR"
	echo -e ${BLUE}"Copying files"${NC}
	cp -a "$TAR_ROOT_DIR"/* $FLASH_DIR/$FLASH_TEMP_DIR/
	mkdir "$FLASH_DIR/$BACKUP_DIR"
	for filename in $FLASH_DIR/$FLASH_TEMP_DIR/*; do
		[ -e "$filename" ] || continue
		filename_base=$(basename -- "$filename")
		mv "$FLASH_DIR/$filename_base" "$FLASH_DIR/$BACKUP_DIR/$filename_base"
		mv "$FLASH_DIR/$FLASH_TEMP_DIR/$filename_base" "$FLASH_DIR/$filename_base"
	done
	rm -rf "$FLASH_DIR/$FLASH_TEMP_DIR"
fi
echo -e ${BLUE}"Remounting flash storage as R/O"${NC}
mount -o remount,ro $FLASH_DIR
echo -e ${BLUE}"Removing temporary files"${NC}
cd $BASE_DIR/
rm -rf $TEMP_DIR/
echo -e ${BLUE}"Update applied successfully!"${NC}
echo -e ${BLUE}"It is strongly recommended to reboot."${NC}
while true; do
	echo -e -n ${BLUE}"Reboot now? [Y]es, [N]o or [C]ancel: "
	read confirmation
	if [ ${confirmation:0:1} == "Y" ]; then
		echo -e "Rebooting... "${NC}
		reboot
		break
	elif [ ${confirmation:0:1} == "N" ] || [ ${confirmation:0:1} == "C" ]; then
		echo -e -n ${NC}
		break
	else
		echo -e "Invalid input \"$confirmation\""${NC}
	fi
done
exit 0
