# import the kodi python modules we are going to use
# see the kodi api docs to find out what functionality each module provides
import xbmc
import xbmcgui
import xbmcaddon
import os				#Used to get OS hostname
import tvh_js			#TVH JSON Interface module
import iso3166			#Convert 2-alpha country to 3-alpha country
import ratinglabels		#Needed to pre-load parental rating labels.
import presets			#External addon definitions for easy updating
						#^^Change the PVR server and client addon names in presets.py

ADDON = xbmcaddon.Addon()
CWD = ADDON.getAddonInfo('path')
I18N = xbmcaddon.Addon().getLocalizedString  #Shortcut for getting translations

CTL_HEADING = 100
CTL_URL = 101

CTL_LIST = 50
CTL_LIST_BACKGROUND = 51

#For the buttons, using the 'language as an example:
#The background colour block is id=901
# The button that is clicked is id=1001
#The text inside that button is id=2001
CTL_BTN_LANGUAGE = 1001
CTL_BTN_HOSTNAME = 1002
CTL_BTN_TUNER_TYPE = 1003
CTL_BTN_COUNTRY = 1004
CTL_BTN_REGION = 1005
CTL_BTN_RATINGS = 1006

CTL_TXT_LANGUAGE = 2001
CTL_TXT_HOSTNAME = 2002
CTL_TXT_TUNER_TYPE = 2003
CTL_TXT_COUNTRY = 2004
CTL_TXT_REGION = 2005
CTL_TXT_RATINGS = 2006

#This is the background colour, the
#text message is +100.
CTL_PROGRESS_LANGUAGE = 3001
CTL_PROGRESS_HOSTNAME = 3002
CTL_PROGRESS_SERVICES = 3003
CTL_PROGRESS_LCNS = 3004
CTL_PROGRESS_CHANNELS = 3005
CTL_PROGRESS_RATINGS = 3006

CTL_BTN_NEXT = 4001
CTL_BTN_BACK = 4002

STATUS_COMPLETE = "greencoal.png"
STATUS_PENDING = "redcoal.png"
STATUS_NOT_SUPPORTED = "charcoal.png"

STR_HEADING = 32000
STR_LANGUAGE = 32001
STR_HOSTNAME = 32002
STR_TUNER_TYPE = 32003
STR_COUNTRY = 32004
STR_REGION = 32005
STR_RATINGS = 32006
STR_PROVIDER = 32007
STR_LONGITUDE = 32008
STR_PROTOCOL = 32009
STR_NOT_SET = 32010
STR_ENABLED = 32011
STR_DISABLED = 32012
STR_EXIT = 32013
STR_RETRY = 32014
STR_NOT_SUPPORTED = 32015
STR_CONFIG_PROCEED = 32016
STR_CONFIG_ABORTED = 32017
STR_NO_ADAPTERS = 32018
STR_CREATE_NET = 32019
STR_SERVICE_SCAN = 32020
STR_SERVICES = 32021
STR_NO_MUX = 32022
STR_COMPLETE = 32023
STR_TIMED_OUT = 32024
STR_CHANNEL_MAP = 32025
STR_CHANNELS_MAPPED = 32026
STR_ABORTED = 32027
STR_LCNS = 32028
STR_FINISHED = 32029
STR_DEFAULT = 32030
STR_ERROR_EXITING = 32031
STR_NO_SERVICES = 32032
STR_CONFIGURE = 32033
STR_CONFIRM_EXIT = 32034
STR_INSTALL_SERVER = 32035
STR_INSTALL_CLIENT = 32036
STR_SET_HOST = 32037
STR_CONFIGURE = 32038




#Convert the Kodi language code to the TVH language code.
#Defaulting to eng_GB.
def getTVHLanguage(kodiLanguage):
	__languageList = {
	"ar":"ara", "bg":"bul", "cs":"cze", "da":"dan", "de":"ger",
	"en":"eng_GB", "es":"spa", "et":"est", "fa":"fas", "fi":"fin",
	"fr":"fre", "he":"heb", "hr":"hrv", "hu":"hun", "it":"ita",
	"ko":"kor", "lv":"lav", "lt":"lit", "nl":"dut", "nb":"nor",
	"pl":"pol", "pt":"por", "ro":"rum", "ru":"rus", "sl":"slv",
	"sk":"slo", "sq":"alb", "sv":"swe", "tr":"tur", "uk":"ukr",
	"zh_tw":"chi", "zh_cn":"chi_CN",
	}
	myLanguage = "eng_GB"
	tmpLanguage = kodiLanguage[:2]
	if tmpLanguage in __languageList:
		myLanguage = __languageList[tmpLanguage.lower()]
	else:
		if kodiLanguage == "zh_tw" or kodiLanguage == "zh-tw":
			myLanguage = "chi"
		if kodiLanguage == "zh_cn" or kodiLanguage == "zh-cn":
			myLanguage = "chi_CN"
	return myLanguage

#A list of the TVH language codes and descriptive names.
#This is not in its own function because it is also used to load the selection list.
#TODO - Get current countries via '/api/language/ui_locale'
#and then merge the 2 lists.  Make this list native names only.
TVH_languageNames = {
"ach":"Acoli", "ady":"Adyghe; Adygei", "alb":"Shqip (Albanian)",
"ara":"اَلْعَرَبِيَّة (Arabic)", "bul":"български (Bulgarian)", "chi":"漢語 (Chinese (traditional))",
"chi_CN":"汉语 (Chinese (simplified))", "cze":"čeština (Czech)",
"dan":"Dansk (Danish)", "dut":"Nederlands (Dutch; Flemish)",
"eng_GB":"English (GB)", "eng_US":"English (US)",
"est":"Eesti Keel (Estonian)", "fas":"فارسی (Persian)",
"fin":"suomi (Finnish)", "fre":"Français (French)",
"ger":"Deutsch (German)", "heb":"עִבְֿרִית‎ (Hebrew)",
"hrv":"Hrvatski (Crotian)", "hun":"Magyar (Hungarian)",
"ita":"Italiano (Italian)", "kor":"한국어 (Korean)",
"lav":"Latviešu (Latvian)", "lit":"Lietuvių (Lithuanian)",
"nor":"Norsk (Norwegian)", "pol":"Polski (Polish)",
"por":"Português (Portuguese)", "rum":"Românește (Romanian; Moldavian; Moldovan)",
"rus":"русский (Russian)", "slo":"Slovenčina (Slovak)",
"slv":"Slovenski (Slovenian)", "spa":"Español (Spanish; Castilian)",
"swe":"Svenska (Swedish)", "tur":"Türkçe (Turkish)", "ukr":"українська (Ukranian)"}




# add a class to create your xml based window
class GUI(xbmcgui.WindowXML):
	# [optional] this function is only needed of you are passing optional data to your window
	def __init__(self, *args, **kwargs):
		# get the optional data and add it to a variable you can use elsewhere in your script
		self.data = kwargs['optional1']
		self.currentField = 0
		self.nextField = 0
		self.supportsRatings = False
		self.__language = ""
		self.__hostname = ""
		self.__tunerType = ""
		self.__country = ""
		self.__region = ""
		self.__ratings = "0"

	def logger(self, logText):
		xbmc.log(logText,level=xbmc.LOGINFO)

	# until now we have a blank window, the onInit function will parse the xml file
	def onInit(self):

		global CONFIG_COMPLETE
		CONFIG_COMPLETE = False

		self.__ipAddress =  xbmc.getIPAddress()
		self.tvh = tvh_js.tvh_js(self.__ipAddress, 9981, "HTSP", "USER", "PASS", self.logger)

		self.__list = self.getControl(CTL_LIST)
		self.__list_background = self.getControl(CTL_LIST_BACKGROUND)
		self.__btn_language = self.getControl(CTL_BTN_LANGUAGE)
		self.__btn_name = self.getControl(CTL_BTN_HOSTNAME)
		self.__btn_tunerType = self.getControl(CTL_BTN_TUNER_TYPE)
		self.__btn_country = self.getControl(CTL_BTN_COUNTRY)
		self.__btn_region = self.getControl(CTL_BTN_REGION)
		self.__btn_ratings = self.getControl(CTL_BTN_RATINGS)

		self.__txt_language = self.getControl(CTL_TXT_LANGUAGE)
		self.__txt_name = self.getControl(CTL_TXT_HOSTNAME)
		self.__txt_tunerType = self.getControl(CTL_TXT_TUNER_TYPE)
		self.__txt_country = self.getControl(CTL_TXT_COUNTRY)
		self.__txt_region = self.getControl(CTL_TXT_REGION)
		self.__txt_ratings = self.getControl(CTL_TXT_RATINGS)

		self.STR_COUNTRY = STR_COUNTRY
		self.STR_REGION = STR_REGION

		self.configuring = False
		self.aborting = False

		#Test to see if this version of TVH supports parental rating labels.
		self.objectClasses = self.tvh.getObjectClasses()
		#self.objectClasses = [] #Make empty for testing.
		if 'ratinglabel' in self.objectClasses:
			xbmc.log("'ratinglabel' object found.",level=xbmc.LOGINFO)
			self.supportsRatings = True
		else:
			xbmc.log("'ratinglabel' object found.  Hiding rating labels.",level=xbmc.LOGINFO)
			self.setStatus(CTL_BTN_RATINGS, STATUS_NOT_SUPPORTED)

		#Test to see if there are any adapters.  Can not proceed past this point.
		self.myAdapters = self.tvh.getAdapters()
		adapterCount = len(self.myAdapters)
		xbmc.log("Adapter count: " + str(adapterCount),level=xbmc.LOGINFO)
		while adapterCount == 0:
			ret = False
			ret = xbmcgui.Dialog().yesno(I18N(STR_HEADING), I18N(STR_NO_ADAPTERS), I18N(STR_RETRY), I18N(STR_EXIT))
			if not ret:
				xbmc.log("No adapters found.  'Retry' selected by user.",level=xbmc.LOGINFO)
				self.myAdapters = self.tvh.getAdapters()
				adapterCount = len(self.myAdapters)
				xbmc.log("Revised adapter count: " + str(adapterCount),level=xbmc.LOGINFO)
			else:
				xbmc.log("No adapters found.  'Exit' selected by user.",level=xbmc.LOGINFO)
				adapterCount = 999
				self.close()

		#Select the view mode in the skin
		xbmc.executebuiltin('Container.SetViewMode(5000)')
		xbmc.log("LANG = " + str(xbmc.getLanguage(xbmc.ISO_639_1)),level=xbmc.LOGINFO)
		xbmc.log("LANG = " + getTVHLanguage(xbmc.getLanguage(xbmc.ISO_639_1)),level=xbmc.LOGINFO)
		#At the time of writing, the Kodi language functions had a bug.
		#xbmc.log("LANG = " + str(xbmc.getLanguage(xbmc.ISO_639_1, True)),level=xbmc.LOGINFO)
		#xbmc.log("LANG = " + str(xbmc.getLanguage(xbmc.ISO_639_2)),level=xbmc.LOGINFO)
		#xbmc.log("LANG = " + str(xbmc.getLanguage(xbmc.ISO_639_2, True)),level=xbmc.LOGINFO)
		#xbmc.log("LANG = " + str(xbmc.getLanguage(xbmc.ENGLISH_NAME)),level=xbmc.LOGINFO)
		#xbmc.log("LANG = " + str(xbmc.getLanguage(xbmc.ENGLISH_NAME, True)),level=xbmc.LOGINFO)

		#Set the top heading of the window with the correct language string
		ctl_title = self.getControl(CTL_HEADING)
		ctl_title.setText(I18N(STR_HEADING))

		#Set the button text
		ctl_next = self.getControl(CTL_BTN_NEXT)
		ctl_next.setLabel(I18N(STR_CONFIGURE))

		ctl_back = self.getControl(CTL_BTN_BACK)
		ctl_back.setLabel(I18N(STR_EXIT))

		#Set the sub-heading to be the URL to TVH
		xbmc.log("IP Address = " + str(self.__ipAddress),level=xbmc.LOGINFO)
		self.setTitleURL(self.__ipAddress)

		#Initialise the list of fields
		self.setLanguage("", getTVHLanguage(xbmc.getLanguage(xbmc.ISO_639_1)))
		self.setHostname(os.uname()[1])
		self.setTunerType(I18N(STR_NOT_SET), "")
		self.setCountry(I18N(STR_NOT_SET), "")
		self.setRegion(I18N(STR_NOT_SET), "")
		if self.supportsRatings:
			self.setRatings(I18N(STR_NOT_SET), "1")
		else:
			self.setRatings(I18N(STR_NOT_SUPPORTED), "0")

		#Give kodi some time to itself.
		xbmc.sleep(100)

		#Set the default focus to the tuner type because the
		#language and hostname are learned from Kodi and the OS.
		self.setFocusId(CTL_BTN_TUNER_TYPE)
		#Activate the tuner type selection
		self.onClick(CTL_BTN_TUNER_TYPE)

	#Set the URL for further manual configuration
	def setTitleURL(self, myIP):
		ctl_host = self.getControl(CTL_URL)
		xbmc.log("Host = " + str(ctl_host.getText()),level=xbmc.LOGINFO)
		ctl_host.setText("[B]http://" + myIP + "[/B]")

	#Set the background colour of the menu item
	def setStatus(self, control, status):
		ctrl = self.getControl(control - 100) #Get the background image, not the button.
		ctrl.setImage(status)

	#Set the selected language
	def setLanguage(self, name, code):
		self.__language = code
		if len(name) == 0:
			if code in TVH_languageNames:
				name = TVH_languageNames[code]
			else:
				name = code
		ctrl = self.getControl(CTL_BTN_LANGUAGE)
		self.__txt_language.setText(I18N(STR_LANGUAGE) + ": [B]" + name + "[/B]")

	#Set the selected hostname
	def setHostname(self, hostname):
		self.__hostname = hostname
		ctl_host = self.getControl(2002)
		ctl_host.setText("Name: [B]" + hostname + "[/B]")

	#Set the selected tuner type
	def setTunerType(self, name, code):
		#TODO - If the tuner type changes, reset the country/region fields
		savedCode = self.__tunerType
		self.__tunerType = code
		self.__txt_tunerType.setText(I18N(STR_TUNER_TYPE) + ": [B]" + name + "[/B]")

		if code == "dvbt":
			self.STR_COUNTRY = STR_COUNTRY
			self.STR_REGION = STR_REGION
		if code == "dvbc":
			self.STR_COUNTRY = STR_COUNTRY
			self.STR_REGION = STR_PROVIDER
		if code == "dvbs":
			self.STR_COUNTRY = STR_LONGITUDE
			self.STR_REGION = STR_PROVIDER
		if code == "atsct":
			self.STR_COUNTRY = STR_COUNTRY
			self.STR_REGION = STR_PROTOCOL
		if code == "atscc":
			self.STR_COUNTRY = STR_COUNTRY
			self.STR_REGION = STR_PROTOCOL
		if code == "isdbt":
			self.STR_COUNTRY = STR_COUNTRY
			self.STR_REGION = STR_REGION

		if code != savedCode:
			xbmc.log("Reset headings",level=xbmc.LOGINFO)
			self.setStatus(CTL_BTN_COUNTRY, STATUS_PENDING)
			self.setCountry(I18N(STR_NOT_SET), "")
			self.setStatus(CTL_BTN_REGION, STATUS_PENDING)
			self.setRegion(I18N(STR_NOT_SET), "")

	#Set the selected country
	def setCountry(self, name, code):
		#TODO - If the tuner type changes, reset the country/region fields
		self.__country = code
		self.__txt_country.setText(I18N(self.STR_COUNTRY) + ": [B]" + name + "[/B]")

	#Set the selected region
	def setRegion(self, name, code):
		#TODO - If the tuner type changes, reset the country/region fields
		self.__region = code
		self.__txt_region.setText(I18N(self.STR_REGION) + ": [B]" + name + "[/B]")
		#Try to extract the country code from the scan file path
		self.eitCountry = ""
		parts = code.split("/")
		if len(parts) > 1:
			xbmc.log("Scan country " + str(parts),level=xbmc.LOGINFO)
			if len(iso3166.lookup(parts[1].upper())) == 3:
				self.eitCountry = iso3166.lookup(parts[1].upper())
				xbmc.log("EIT country " + self.eitCountry,level=xbmc.LOGINFO)

	#Set the parental ratings
	def setRatings(self, name, code):
		self.__ratings = code
		self.__txt_ratings.setText(I18N(STR_RATINGS) + ": [B]" + name + "[/B]")

	#Load the list with available languages and pre-select current language
	def loadListLanguages(self):
		counter = 0
		selected = -1
		self.__list.reset()
		self.listitems = []
		for lang in TVH_languageNames:
			listitem = xbmcgui.ListItem(TVH_languageNames[lang], lang)
			if lang == self.__language:
				selected = counter
			counter = counter + 1
			self.listitems.append(listitem)
		self.__list.addItems(self.listitems)
		self.__list.controlLeft(self.__btn_name)
		self.__list.controlRight(self.__btn_tunerType)
		if selected != -1:
			self.__list.selectItem(selected)
		self.__list.setVisible(True)
		self.setFocusId(CTL_LIST)

	#Load the list with available tuner types and pre-select current type
	def loadListTunerTypes(self):
		counter = 0
		selected = -1
		self.myAdapters = self.tvh.getAdapters()
		if len(self.myAdapters) == 0:
			xbmc.log("No adapters found",level=xbmc.LOGINFO)
			#TODO - SCREEN DIALOGUE
		else:
			self.__list.reset()
			self.listitems = []
			for adpt in self.myAdapters:
				xbmc.log("Adapter type: '" + adpt + "'",level=xbmc.LOGINFO)
				listitem = xbmcgui.ListItem(self.tvh.getNetworkType(adpt).upper(), adpt)
				if adpt == self.__tunerType:
					selected = counter
				counter = counter + 1
				self.listitems.append(listitem)
			#Manually add extra tuner types for testing only.
			#listitem = xbmcgui.ListItem("DVB-S", "dvbs")
			#self.listitems.append(listitem)
			#listitem = xbmcgui.ListItem("ATSC-T", "atsct")
			#self.listitems.append(listitem)
			#listitem = xbmcgui.ListItem("ATSC-C", "atscc")
			#self.listitems.append(listitem)
			#listitem = xbmcgui.ListItem("ISDB-T", "isdbt")
			#self.listitems.append(listitem)

			self.__list.addItems(self.listitems)
			self.__list.controlLeft(self.__btn_tunerType)
			self.__list.controlRight(self.__btn_country)
			if selected != -1:
				self.__list.selectItem(selected)
			self.__list.setVisible(True)
			self.setFocusId(CTL_LIST)

	#Load the list with available countries and pre-select current one
	#The countries come from the tuner type, dvb-t, dvb-c, etc.
	def loadListCountries(self):
		counter = 0
		selected = -1
		#scanFiles = tvh.getScanFiles("dvbs", 28.2)
		self.__list.reset()
		self.listitems = []
		self.myScanfiles = self.tvh.getScanFiles(self.__tunerType)
		if len(self.myScanfiles) == 0:
			xbmc.log("No scan files",level=xbmc.LOGINFO)
			#TODO - DIALOGUE
		else:
			for country in self.myScanfiles:
				listitem = xbmcgui.ListItem(country, country)
				if country == self.__country:
					selected = counter
				counter = counter + 1
				self.listitems.append(listitem)
			self.__list.addItems(self.listitems)
			self.__list.controlLeft(self.__btn_country)
			self.__list.controlRight(self.__btn_region)
			if selected != -1:
				self.__list.selectItem(selected)
			self.__list.setVisible(True)
			self.setFocusId(CTL_LIST)

	#Load the list with available regions for the selected country
	#and pre-select current one
	def loadListRegions(self):
		counter = 0
		selected = -1
		self.__list.reset()
		self.listitems = []
		#self.myScanfiles = self.tvh.getScanFiles(self.__tunerType)
		if len(self.myScanfiles) == 0:
			xbmc.log("No scan files",level=xbmc.LOGINFO)
			TODO - DIALOGUE
		else:
			for region in self.myScanfiles[self.__country]['regions']:
				listitem = xbmcgui.ListItem(region['name'], region['path'])
				if region['path'] == self.__region:
					selected = counter
				counter = counter + 1
				self.listitems.append(listitem)
			self.__list.addItems(self.listitems)
			self.__list.controlLeft(self.__btn_region)
			self.__list.controlRight(self.__btn_ratings)
			if selected != -1:
				self.__list.selectItem(selected)
			self.__list.setVisible(True)
			self.setFocusId(CTL_LIST)

	#Load enables/disabled for parental rating labels
	def loadListRatings(self):
		if self.supportsRatings:
			counter = 0
			selected = -1
			self.__list.reset()
			self.listitems = []
			listitem = xbmcgui.ListItem(I18N(STR_DISABLED), "0")
			self.listitems.append(listitem)
			listitem = xbmcgui.ListItem(I18N(STR_ENABLED), "1")
			self.listitems.append(listitem)

			self.__list.addItems(self.listitems)
			self.__list.controlLeft(self.__btn_ratings)

			if self.__ratings == "0":
				self.__list.selectItem(0)
			if self.__ratings == "1":
				self.__list.selectItem(1)

			self.__list.setVisible(True)
			self.setFocusId(CTL_LIST)
		else:
			self.setFocusId(CTL_BTN_NEXT)

	#Update the progress status message
	def setProgress(self, control, background, text):
		#Updatew the background colour
		ctrl = self.getControl(control)
		ctrl.setImage(background)
		#Update the text message
		ctrl = self.getControl(control + 100)
		ctrl.setText(text)


	#Check the settings and then configure TVH
	def doConfiguration(self):
		global CONFIG_COMPLETE
		CONFIG_COMPLETE = False
		xbmc.log("===== TVH CONFIGURATION =====",level=xbmc.LOGINFO)
		xbmc.log("          LANGUAGE = '" + self.__language + "'",level=xbmc.LOGINFO)
		xbmc.log("          HOSTNAME = '" + self.__hostname + "'",level=xbmc.LOGINFO)
		xbmc.log("        TUNER TYPE = '" + self.__tunerType + "'",level=xbmc.LOGINFO)
		xbmc.log("           COUNTRY = '" + self.__country + "'",level=xbmc.LOGINFO)
		xbmc.log("            REGION = '" + self.__region + "'",level=xbmc.LOGINFO)
		xbmc.log("  PARENTAL RATINGS = '" + self.__ratings + "', '" + self.eitCountry + "'",level=xbmc.LOGINFO)

		ret = False
		ret = xbmcgui.Dialog().yesno(I18N(STR_HEADING), I18N(STR_CONFIG_PROCEED))
		xbmc.log("          RESPONSE = '" + str(ret) + "'",level=xbmc.LOGINFO)
		if ret:
			self.configuring = True
			#self.__list.reset()
			#self.listitems = []
			#self.__list.setVisible(True)

			#Hide the listbox.
			self.__list.setVisible(False)
			self.__list_background.setVisible(False)

			#Set the TVH UI language
			xbmc.log("Setting language = '" + self.__language + "'",level=xbmc.LOGINFO)
			retLang = self.tvh.setGUILanguage(self.__language)
			#xbmc.log(str(retLang),level=xbmc.LOGINFO)
			#self.__list.addItem(xbmcgui.ListItem(I18N(STR_LANGUAGE) + ": [B]" + I18N(STR_COMPLETE) + "[/B]"))
			self.setProgress(CTL_PROGRESS_LANGUAGE, STATUS_COMPLETE, I18N(STR_LANGUAGE) + ": [B]" + I18N(STR_COMPLETE) + "[/B]")

			#Set the TVH server name
			xbmc.log("Setting server name = '" + self.__hostname + "'",level=xbmc.LOGINFO)
			retName = self.tvh.setServerName(self.__hostname)
			#xbmc.log(str(retName),level=xbmc.LOGINFO)
			#self.__list.addItem(xbmcgui.ListItem(I18N(STR_HOSTNAME) + ": [B]" + I18N(STR_COMPLETE) + "[/B]"))

			self.setProgress(CTL_PROGRESS_HOSTNAME, STATUS_COMPLETE, I18N(STR_HOSTNAME) + ": [B]" + I18N(STR_COMPLETE) + "[/B]")

			#TODO - Set the name in the pvr.hts addon
			#^^^^ The python API does not permit this for
			#     addon 'instances'.

			#Get ready to create a network and load the muxes.
			#Build a list of UUIDs for the selected tuner type
			adapterUUIDs = []
			for leaf in self.myAdapters[self.__tunerType]:
				adapterUUIDs.append(leaf['uuid'])
			#xbmc.log("Leaf adapter UUIDs = '" + str(adapterUUIDs) + "'",level=xbmc.LOGINFO)

			#Create a new network and associated muxes
			netMuxes = []
			newNetwork, netMuxes = self.tvh.networkCreate(I18N(STR_DEFAULT) + "-" + self.tvh.getNetworkType(self.__tunerType).upper(), self.__tunerType, adapterUUIDs, self.__region)
			newNetworkUUID = newNetwork['uuid']
			#xbmc.log(str(newNetwork),level=xbmc.LOGINFO)
			#xbmc.log(str(netMuxes),level=xbmc.LOGINFO)

			#Load an dict to keep track of outstanding muxes
			#As a mux is successfully scanned, it will be removed
			muxList = {}
			muxNames = {}
			for mux in netMuxes:
				if mux['uuid'] not in muxList:
					muxList[mux['uuid']] = 1
					muxNames[mux['uuid']] = mux['name']
					xbmc.log("Mux '" + mux['name'] + "' created, uuid = '" + mux['uuid'] + "'",level=xbmc.LOGINFO)
			#xbmc.log(str(muxList),level=xbmc.LOGINFO)
			#xbmc.log(str(len(muxList)),level=xbmc.LOGINFO)

			#print("New Network UUID:", newNetworkUUID, "mux count", len(netMuxes))
			xbmc.log("New network UUID = '" + newNetworkUUID + "' Mux count = '" + str(len(netMuxes)) + "'",level=xbmc.LOGINFO)
			#self.__list.addItem(xbmcgui.ListItem(I18N(STR_CREATE_NET) + ": [B]" + I18N(STR_COMPLETE) + "[/B]"))

			#For the newly created network, add it to each of the selected adapters.
			for adptr in adapterUUIDs:
				#print("Adding network '" + newNetworkUUID + "' to adapter '" + adptr + "'")
				self.tvh.networkLinkAdapter(newNetworkUUID, adptr)

			#Start a scan for services on each mux of a network
			#self.__list.addItem(xbmcgui.ListItem(I18N(STR_SERVICE_SCAN) + ": [B]Started[/B]"))
			#self.__list.addItem(xbmcgui.ListItem(I18N(STR_SERVICE_SCAN) + ": [B]Progress[/B]"))
			#self.__list.addItem(xbmcgui.ListItem(I18N(STR_SERVICE_SCAN) + ":"))
			#self.__list.addItem(xbmcgui.ListItem(I18N(STR_SERVICE_SCAN) + ": [B]0%[/B]"))
			self.tvh.networkPerformScan(newNetworkUUID)
			#The scan can't be allowed to continue until every service
			#has an LCN.  So allow for 3 attempts at 30 seconds per mux
			#and if not every service has an LCN by then, give up.
			#If the total scan time would exceed 2 minutes,
			#then cap it at 2 minutes.
			#NOTE: This cap may have to be revisited for Satellite systems.

			#exitCount = (len(netMuxes) * 30 * 3) // len(adapterUUIDs)
			exitCount = (len(netMuxes) * presets.SECONDS_PER_MUX * presets.LCN_RETRIES) // len(adapterUUIDs)
			xbmc.log("Proposed max runtime " + str(exitCount), level=xbmc.LOGINFO)
			#if exitCount > 120:
			#	exitCount = 120
			maxExit = exitCount

			runFlag = True
			scanTimedOut = False

			if len(netMuxes) == 0:
				xbmc.log("No muxes found", level=xbmc.LOGINFO)
				xbmcgui.Dialog().ok(I18N(STR_HEADING), I18N(STR_NO_MUX))
				self.close()
				return

			rescanCount = {}  #Used to keep track of how many times a mux has scanned for LCNs

			#Do a loop until all services have an LCN or we time-out.
			while runFlag and (not xbmc.Monitor().abortRequested()) and (not self.aborting):
				resp, lcnMux, failedMux = self.tvh.networkGetStatus(newNetworkUUID)
				#xbmc.log(str(resp), level=xbmc.LOGINFO)
				#xbmc.log(str(lcnMux), level=xbmc.LOGINFO)
				#xbmc.log("Failed: " + str(failedMux), level=xbmc.LOGINFO)
				xbmc.log("Pending scans: " + str(resp['remaining']) + ", service count: " + str(resp['serviceCount']) + ", LCN count: " + str(resp['lcns']) + ", failed scans: " + str(len(failedMux)), level=xbmc.LOGINFO)
				misMatch = resp['serviceCount'] - resp['lcns']
				if (resp['serviceCount'] != 0) and (misMatch == 0) and (resp['remaining'] == 0):
					exitCount = 0
					runFlag = False
				if exitCount > 0 and runFlag:  #We still have time remaining
					#self.__list.removeItem(4)
					#self.__list.removeItem(3)
					#self.__list.addItem(xbmcgui.ListItem(I18N(STR_SERVICE_SCAN) + ": [B]{:.0f}%[/B]".format(100-((exitCount / maxExit)*100))))
					#self.__list.addItem(xbmcgui.ListItem(I18N(STR_SERVICE_SCAN) + ": [B]" + I18N(STR_SERVICES) + ": {}, LCNs {}.[/B]".format(resp['serviceCount'],resp['lcns'])))

					self.setProgress(CTL_PROGRESS_SERVICES, STATUS_NOT_SUPPORTED, I18N(STR_SERVICE_SCAN) + ": [B]{:.0f}%[/B]".format(100-((exitCount / maxExit)*100)))
					self.setProgress(CTL_PROGRESS_LCNS, STATUS_NOT_SUPPORTED, I18N(STR_SERVICES) + ": [B]{}[/B], ".format(resp['serviceCount']) + I18N(STR_LCNS) + ": [B]{}[/B]".format(resp['lcns']))

					xbmc.log("LCN shortfall " + str(misMatch) + ". Max seconds remaining " + str(exitCount), level=xbmc.LOGINFO)
					#xbmc.sleep(1000)
					xbmc.Monitor().waitForAbort(1)  #Sleep for 1 second.
					exitCount = exitCount - 1

					if exitCount == 0 or xbmc.Monitor().abortRequested():
						runFlag = False
						scanTimedOut = True
						xbmc.log("Service scan timed-out", level=xbmc.LOGINFO)
					else:
						if resp['remaining'] == 0:  #If all of the muxes have finished scanning
							if misMatch != 0:  #If the number of LCNs does not match the number of services

								xbmc.log("muxList before " + str(len(muxList)), level=xbmc.LOGINFO)
								#Remove muxes that faile to scan from the outstanding mux list
								for failed in failedMux:
									if failed in muxList:
										xbmc.log("Failing mux '" + failed + "' (" + muxNames[failed] + ") scans (" + str(muxList[failed]) + ")", level=xbmc.LOGINFO)
										del muxList[failed]

								#Compare the reported muxes to the outstanding muxes and
								#remove muxes from the outstanding luist that now have services
								for lm in lcnMux:
									if lm in muxList:
										if (lcnMux[lm]['count'] != 0) or (muxList[lm] >= 3):
											xbmc.log("De-queueing mux '" + lm + "' (" + muxNames[lm] + ") scans (" + str(muxList[lm]) + ") count (" + str(lcnMux[lm]['count']) + ")", level=xbmc.LOGINFO)
											del muxList[lm]
										else:
											muxList[lm] = muxList[lm] + 1

								xbmc.log("muxList after " + str(len(muxList)), level=xbmc.LOGINFO)
								#xbmc.log(str(muxList), level=xbmc.LOGINFO)
								if len(muxList) == 0:
									runFlag = False
								else:
									for mx in muxList:
										#xbmc.log("muxList '" + mx + "' scans = " + str(muxList[mx]), level=xbmc.LOGINFO)
										self.tvh.muxPerformScan(mx)
										xbmc.log("Rescheduling mux '" + str(mx) + "' (" + muxNames[mx] + ") (" + str(muxList[mx]) + ")", level=xbmc.LOGINFO)

							else:
								runFlag = False
				else:
					#Kodi status update message
					xbmc.log("LCN scan complete: services " + str(resp['serviceCount']) + " LCNs " + str(resp['lcns']), level=xbmc.LOGINFO)
					#self.__list.removeItem(4)
					#self.__list.removeItem(3)
					if scanTimedOut:
						#self.__list.addItem(xbmcgui.ListItem(I18N(STR_SERVICE_SCAN) + ": [B]" + I18N(STR_TIMED_OUT) + "[/B]"))
						self.setProgress(CTL_PROGRESS_SERVICES, STATUS_NOT_SUPPORTED, I18N(STR_SERVICE_SCAN) + ": [B]" + I18N(STR_TIMED_OUT) + "[/B]")
					else:
						#self.__list.addItem(xbmcgui.ListItem(I18N(STR_SERVICE_SCAN) + ": [B]" + I18N(STR_COMPLETE) + "[/B]"))
						self.setProgress(CTL_PROGRESS_SERVICES, STATUS_COMPLETE, I18N(STR_SERVICE_SCAN) + ": [B]" + I18N(STR_COMPLETE) + "[/B]")
						self.setProgress(CTL_PROGRESS_LCNS, STATUS_COMPLETE, I18N(STR_SERVICES) + ": [B]{}[/B], ".format(resp['serviceCount']) + I18N(STR_LCNS) + ": [B]{}[/B]".format(resp['lcns']))
					#self.__list.addItem(xbmcgui.ListItem(I18N(STR_SERVICE_SCAN) + ": [B]" + I18N(STR_SERVICES) + ": {}, LCNs {}[/B]".format(resp['serviceCount'],resp['lcns'])))
					#self.setProgress(CTL_PROGRESS_LCNS, STATUS_NOT_SUPPORTED, I18N(STR_SERVICES) + ": [B]{}[/B], ".format(resp['serviceCount']) + I18N(STR_LCNS) + ": [B]{}[/B]".format(resp['lcns']))

			#If the user has interrupted the scanning process, exit.
			#Note: TVH will still keep scanning until it is finished,
			#it may find some services, however, these will not be mapped
			#to channels.
			if self.aborting:
				CONFIG_COMPLETE = False
				self.close()
				return

			#If there have been no services found, display an error and exit.
			if resp['serviceCount'] == 0:
				xbmcgui.Dialog().ok(I18N(STR_HEADING), I18N(STR_NO_SERVICES))
				CONFIG_COMPLETE = False
				self.close()
				return

			#If this PVR is Australian, patch the OTA EPG Genres
			if self.eitCountry == "AUS":
				xbmc.log("Patching Australian OTA genre codes.", level=xbmc.LOGINFO)
				self.tvh.setOTAGenreTranslate("192=20\n208=16\n224=35")


			#Enable Parental Rating Labels if required and supported
			#This is done before mapping channels so that when
			#the EPG is grabbed, the rating labels can be recognised.
			if self.supportsRatings:
				xbmc.log("Parental Rating Labels supported", level=xbmc.LOGINFO)
				if self.__ratings == "0":
					self.tvh.setParentalRating(False)
					xbmc.log("Parental Rating Labels disabled", level=xbmc.LOGINFO)
				if self.__ratings == "1":
					self.tvh.setParentalRating(True)
					xbmc.log("Parental Rating Labels enabled", level=xbmc.LOGINFO)
					labelLocation = CWD + "resources/ratingicons/"
					myLabels = ratinglabels.findLabels(self.eitCountry, labelLocation, "file://" + labelLocation)
					xbmc.log("Parental Rating Labels for '" + self.eitCountry + "' found: " + str(len(myLabels)), level=xbmc.LOGINFO)
					if len(myLabels) != 0:
						for label in myLabels:
							rlRet = self.tvh.ratingLabelCreate(label)
					self.setProgress(CTL_PROGRESS_RATINGS, STATUS_COMPLETE, I18N(STR_RATINGS) + ": [B]" + I18N(STR_COMPLETE) + "[/B]")
			else:
				xbmc.log("Parental Rating Labels not supported", level=xbmc.LOGINFO)

			#Map channels to services
			#self.__list.addItem(xbmcgui.ListItem(I18N(STR_CHANNEL_MAP) + ": [B]0[/B]"))
			chList, tagList = self.tvh.networkMapServices(newNetworkUUID, True)
			#self.__list.addItem(xbmcgui.ListItem(I18N(STR_CHANNELS_MAPPED) + ": [B]{}[/B]".format(len(chList))))
			self.setProgress(CTL_PROGRESS_CHANNELS, STATUS_COMPLETE, I18N(STR_CHANNELS_MAPPED) + ": [B]{}[/B]".format(len(chList)))
			xbmc.log("Channels mapped = " + str(len(chList)), level=xbmc.LOGINFO)
			xbmc.log("Tags created = " + str(len(tagList)), level=xbmc.LOGINFO)

			xbmc.log("TVHeadEnd PVR Quick Start Complete!", level=xbmc.LOGINFO)
			xbmcgui.Dialog().ok(I18N(STR_HEADING), I18N(STR_FINISHED))
			CONFIG_COMPLETE = True
			self.close()
			return

		else:
			xbmcgui.Dialog().ok(I18N(STR_HEADING), I18N(STR_ABORTED))
			self.close()
			return

	def onFocus(self, controlID):
		xbmc.log("ONFOCUS = " + str(controlID),level=xbmc.LOGINFO)

	#Test for a click from the controls on the screen and
	#perform the required actions.
	def onClick(self, controlID):
		xbmc.log("GOT CLICK = " + str(controlID),level=xbmc.LOGINFO)
		#This is the list of options.  The actions can change based on
		#which field the options relate to.
		if controlID == CTL_LIST:
			#list = self.getControl(controlID)
			#xbmc.log("GOT CLICK = " + str(type(list)),level=xbmc.LOGINFO)
			#xbmc.log("GOT CLICK = " + str(list.__dir__()),level=xbmc.LOGINFO)
			label1 = ""		#This will normally have the actual displayed value
			label2 = ""		#This will normally have the key for the displayed value
			sel = self.__list.getSelectedPosition()
			if sel != -1:
				label1 = self.listitems[sel].getLabel()
				label2 = self.listitems[sel].getLabel2()
				xbmc.log("SELECTED = " + str(sel) + " " + label1 + " " + label2,level=xbmc.LOGINFO)
			#xbmc.log("GOT CLICK = " + str(type(sel)),level=xbmc.LOGINFO)
			#xbmc.log("GOT CLICK = " + str(sel),level=xbmc.LOGINFO)
			#xbmc.log("GOT CLICK = " + str(sel.__dir__()),level=xbmc.LOGINFO)
			#We got a click from within the list

			if self.currentField == CTL_BTN_LANGUAGE:    #Language option
				self.setLanguage(label1, label2)
				self.setFocusId(CTL_BTN_HOSTNAME)
				self.setFocusId(CTL_BTN_TUNER_TYPE)
				self.__list.setVisible(False)

			if self.currentField == CTL_BTN_TUNER_TYPE:    #Tuner type option
				self.setTunerType(label1, label2)
				self.setStatus(CTL_BTN_TUNER_TYPE, STATUS_COMPLETE)
				self.setFocusId(CTL_BTN_COUNTRY)
				self.__list.setVisible(False)

			if self.currentField == CTL_BTN_COUNTRY:    #Country option
				self.setCountry(label1, label2)
				self.setStatus(CTL_BTN_COUNTRY, STATUS_COMPLETE)
				self.setFocusId(CTL_BTN_REGION)
				self.__list.setVisible(False)

			if self.currentField == CTL_BTN_REGION:    #Region option
				self.setRegion(label1, label2)
				self.setStatus(CTL_BTN_REGION, STATUS_COMPLETE)
				self.setFocusId(CTL_BTN_RATINGS)
				self.__list.setVisible(False)

			if self.currentField == CTL_BTN_RATINGS:    #Parental rating option
				self.setRatings(label1, label2)
				self.setStatus(CTL_BTN_RATINGS, STATUS_COMPLETE)
				self.setFocusId(CTL_BTN_NEXT)
				self.__list.setVisible(False)

			#Progress to next field
			if self.nextField != 0:
				#TODO - Only do this if were are still in guided mode.
				xbmc.log("GOING NEXT " + str(self.nextField),level=xbmc.LOGINFO)
				self.onClick(self.nextField)

		#Language field clicked
		if controlID == CTL_BTN_LANGUAGE:
			self.currentField = controlID
			self.nextField = controlID + 1
			self.loadListLanguages()

		#Host name field clicked
		if controlID == CTL_BTN_HOSTNAME:
			ret =  xbmcgui.Dialog().input(I18N(STR_SET_HOST), self.__hostname, xbmcgui.INPUT_ALPHANUM)
			if len(ret) == 0:
				ret = self.__hostname
			xbmc.log("NAME = " + ret,level=xbmc.LOGINFO)
			self.setHostname(ret)
			self.__hostname = ret

		#Tuner type field clicked
		if controlID == CTL_BTN_TUNER_TYPE:
			self.currentField = controlID
			self.nextField = controlID + 1
			self.loadListTunerTypes()

		#Country field clicked
		if controlID == CTL_BTN_COUNTRY:
			self.currentField = controlID
			self.nextField = controlID + 1
			self.loadListCountries()

		#Region field clicked
		if controlID == CTL_BTN_REGION:
			self.currentField = controlID
			self.nextField = controlID + 1
			self.loadListRegions()

		#Parental rating field clicked
		if controlID == CTL_BTN_RATINGS:
			self.currentField = controlID
			self.nextField = controlID + 1
			self.loadListRatings()

		#'Next' button clicked
		#Test to see if all of the fields are valid and then proceed
		#to the actual configuration.
		if controlID == CTL_BTN_NEXT:
			#self.currentField = controlID
			#self.nextField = controlID + 1

			if self.__language == "":
				self.onClick(CTL_BTN_COUNTRY)
				return

			if self.__hostname == "":
				self.onClick(CTL_BTN_HOSTNAME)
				return

			if self.__tunerType == "":
				self.onClick(CTL_BTN_TUNER_TYPE)
				return

			if self.__country == "":
				self.onClick(CTL_BTN_COUNTRY)
				return

			if self.__region == "":
				self.onClick(CTL_BTN_REGION)
				return

			self.doConfiguration()

		#'Exit' button clicked
		if controlID == CTL_BTN_BACK:
			ret = False
			ret = xbmcgui.Dialog().yesno(I18N(STR_HEADING), I18N(STR_CONFIRM_EXIT))
			if ret:
				if self.configuring:
					self.aborting = True
				else:
					self.close()
					return

	#This takes over full control of the keyboard and
	#stops keys like ESC from working.
	#Perhaps this can be made to work in the future,
	#for now, it will remain disabled.
	# def onAction(self, action):
		# # onAction in WindowXML works same as on a Window or WindowDialog its for keypress/controller buttons etc
		# # This function has been implemented and works
		# # buttonCode =  action.getButtonCode()
		# # actionID   =  action.getId()
		# # Action 92 is the back button
		# xbmc.log("GOT ACTION " + str(action.getId()),level=xbmc.LOGINFO)
		# if action == 14:   #Stop RCU key
			# self.onClick(CTL_BTN_BACK)
		# if action == 79:   #Play/Pause RCU key
			# self.onClick(CTL_BTN_NEXT)


# this is the entry point of your addon, execution of your script will start here
if (__name__ == '__main__'):
	xbmc.log("*** Starting TVHeadEnd Quick Start ***", level=xbmc.LOGINFO)

	#The Kodi system name could also be set, however, this is outside of the scope of this addon.
	#response = xbmc.executeJSONRPC('{"jsonrpc":"2.0","id":1,"method":"Settings.getSettingValue","params":{"setting":"services.devicename"}}')
	#xbmc.log(str(response), level=xbmc.LOGINFO)
	#response = xbmc.executeJSONRPC('{"jsonrpc":"2.0","id":1,"method":"Settings.setSettingValue","params":{"setting":"services.devicename", "value": "DMCTEST"}}')
	#response = xbmc.executeJSONRPC('{"jsonrpc":"2.0","id":1,"method":"Settings.getSettingValue","params":{"setting":"services.devicename"}}')
	#xbmc.log(str(response), level=xbmc.LOGINFO)

	#response = xbmc.executeJSONRPC('{"jsonrpc":"2.0","id":1,"method":"Addons.GetAddons"}')
	#xbmc.log(str(response), level=xbmc.LOGINFO)

	global CONFIG_COMPLETE
	CONFIG_COMPLETE = False

	#Get the name of the default server and client addons
	#These are stored in the presets.py file for easier updating should
	#the names (eg service.tvheadendNN) change.
	SERVER_ADDON = presets.SERVER_ADDON
	CLIENT_ADDON = presets.CLIENT_ADDON

	#Blindly try to enable the addons just in case they are installed but disabled.
	#This will fail silently if the addons are not installed.
	if len(SERVER_ADDON) == 0:
		SERVER_ADDON = "service.tvheadend42"

	response = xbmc.executeJSONRPC('{"jsonrpc":"2.0","method":"Addons.SetAddonEnabled","params":{"addonid":"' + SERVER_ADDON + '","enabled":true},"id":1}')
	xbmc.log("Attempting to enable server addon '" + SERVER_ADDON +"':" + str(response), level=xbmc.LOGINFO)

	serverInstalled = xbmc.getCondVisibility("System.HasAddon(" + SERVER_ADDON + ")")
	serverEnabled = xbmc.getCondVisibility("System.AddonIsEnabled(" + SERVER_ADDON + ")")
	xbmc.log("SERVER_ADDON: '" + str(SERVER_ADDON) + "', installed: '" + str(serverInstalled) + "', enabled: '" + str(serverEnabled) + "'", level=xbmc.LOGINFO)

	if len(CLIENT_ADDON) == 0:
		CLIENT_ADDON = "pvr.hts"

	response = xbmc.executeJSONRPC('{"jsonrpc":"2.0","method":"Addons.SetAddonEnabled","params":{"addonid":"' + CLIENT_ADDON + '","enabled":true},"id":1}')
	xbmc.log("Attempting to enable client addon '" + CLIENT_ADDON +"':" + str(response), level=xbmc.LOGINFO)

	xbmc.log("CLIENT_ADDON: '" + str(CLIENT_ADDON) + "'", level=xbmc.LOGINFO)

	clientInstalled = xbmc.getCondVisibility("System.HasAddon(" + CLIENT_ADDON + ")")
	clientEnabled = xbmc.getCondVisibility("System.AddonIsEnabled(" + CLIENT_ADDON + ")")
	xbmc.log("CLIENT_ADDON = '" + str(CLIENT_ADDON) + "', installed: '" + str(clientInstalled) + "', enabled: '" + str(clientEnabled) + "'", level=xbmc.LOGINFO)

	#If for whatever reason the addons are not enabled, try to install them.
	#This used to enable the addon, but now it seems not to.
	#I probably did somethinf wrong somewhere.
	if not serverEnabled:
		serverInstalled = False
	if not clientEnabled:
		clientInstalled = False

	#If either TVH server or client is not installed, try to install them.
	if (not serverInstalled) or (not clientInstalled):
		COUNT_MAX = 120
		prog = xbmcgui.DialogProgress()
		prog.create(I18N(STR_HEADING), I18N(STR_INSTALL_SERVER))
		if not serverInstalled:
			xbmc.log("Installing TVHeadEnd server add-on.",level=xbmc.LOGINFO)
			try:
				xbmc.executebuiltin('InstallAddon(' + SERVER_ADDON + ')')
			except Exception as err:
				#Note: This call never fails because it launches another process.
				#      It is that other process that will fail of the addon is not found.
				xbmc.log("TVHeadEnd server add-on installation FAILED.",level=xbmc.LOGINFO)

			countDown = COUNT_MAX
			while (countDown != 0) and (not serverInstalled):
				prog.update(int((COUNT_MAX - countDown) / COUNT_MAX * 100))
				#https://kodi.wiki/view/Window_IDs
				flag = xbmc.getCondVisibility('Window.IsVisible(yesnodialog)')
				xbmc.log("Dialogue found: " + str(flag),level=xbmc.LOGINFO)
				if flag:
					xbmc.executebuiltin('SendClick(11)') #Send click to 'yes' button.
				xbmc.sleep(500)  #Snooze for 0.5 seconds
				try:
					serverInstalled = True
					tvh_srv = xbmcaddon.Addon(SERVER_ADDON).getAddonInfo("type")
				except Exception as err:
					#I thought that I could trap the error, and I can, but
					#the error is the same if the addon does not exist at all
					#or has not yet finished installing.  We just need to wait
					#for the timeout to expire to see if the installation went OK.
					serverInstalled = False

				xbmc.log("Installing add-ons - Countdown: " + str(countDown) + ", " + SERVER_ADDON + ": " + str(serverInstalled) + ", " + CLIENT_ADDON + ": " + str(clientInstalled),level=xbmc.LOGINFO)
				countDown = countDown -1
				if prog.iscanceled():
					serverInstalled = False
					countDown = 0

		if (not clientInstalled) and serverInstalled:
			xbmc.log("Installing TVHeadEnd client add-on.",level=xbmc.LOGINFO)
			prog.update(0, I18N(STR_INSTALL_CLIENT))
			try:
				xbmc.executebuiltin('InstallAddon(' + CLIENT_ADDON + ')')
			except:
				xbmc.log("TVHeadEnd client add-on installation FAILED.",level=xbmc.LOGINFO)

			countDown = COUNT_MAX
			while (countDown != 0) and (not clientInstalled):
				prog.update(int((COUNT_MAX - countDown) / COUNT_MAX * 100))
				flag = xbmc.getCondVisibility('Window.IsVisible(yesnodialog)')
				xbmc.log("Dialogue found: " + str(flag),level=xbmc.LOGINFO)
				if flag:
					xbmc.executebuiltin('SendClick(11)')
				xbmc.sleep(500)  #Snooze for 0.5 seconds
				try:
					clientInstalled = True
					tvh_clnt = xbmcaddon.Addon(CLIENT_ADDON).getAddonInfo("type")
				except:
					clientInstalled = False
				xbmc.log("Installing add-ons - Countdown: " + str(countDown) + ", " + SERVER_ADDON + ": " + str(serverInstalled) + ", " + CLIENT_ADDON + ": " + str(clientInstalled),level=xbmc.LOGINFO)
				countDown = countDown -1
				if prog.iscanceled():
					clientInstalled = False
					countDown = 0
		#Close the progress dialogue.
		prog.close()


	#Tell the user the bad news. :-(
	if (not serverInstalled) or (not clientInstalled):
		xbmc.log("Error installing server or client addon.  Existing.",level=xbmc.LOGINFO)
		xbmcgui.Dialog().ok(I18N(STR_HEADING), I18N(STR_ERROR_EXITING))
	#Show the quickstart screen, we're good to go.
	else:
		ui = GUI('main_screen.xml', CWD, 'default', '1080i', True, optional1='some data')
		ui.doModal()
		del ui
		#Load the EPG guide view
		if CONFIG_COMPLETE:
			xbmc.log("TVHeadEnd configuration complete, loading EPG Grid.",level=xbmc.LOGINFO)
			xbmc.executebuiltin('ActivateWindow("TVGuide")')

