#
#Object wrapper for interfacing with tvheadend via HTTP/JSON
#
#Copyright (c) DeltaMikeCharlie 2024
#

import json
import urllib
from urllib import request, parse

class tvh_js:

	__hostName = ""
	__httpPort = 0
	__username = ""
	__password = ""

	#Add a new network type, like ATSC, here.
	#'adapter' is the object class for listing adapters.
	#'network' is the object class for creating a network.
	#'scanfile' is the scanfile type.
	__networkTypes = {
	"dvbt": {"adapter": "linuxdvb_frontend_dvbt", "network": "dvb_network_dvbt", "scanfile": "dvb-t"},
	"dvbc": {"adapter": "linuxdvb_frontend_dvbc", "network": "dvb_network_dvbc", "scanfile": "dvb-c"},
	"dvbs": {"adapter": "linuxdvb_frontend_dvbs", "network": "dvb_network_dvbs", "scanfile": "dvb-s"},
	"atsct": {"adapter": "linuxdvb_frontend_atsct", "network": "dvb_network_atsc_t", "scanfile": "atsc-t"},
	"atscc": {"adapter": "linuxdvb_frontend_atscc", "network": "dvb_network_atsc_c", "scanfile": "atsc-c"},
	"isdbt": {"adapter": "linuxdvb_frontend_isdbt", "network": "dvb_network_isdb_t", "scanfile": "isdb-t"}
	}


	#Initialise the object
	def __init__(self, tvhHOST, tvhHTTP, tvhHTSP, tvhUsername, tvhPassword, logger):
		#print("Creating TVH object:", tvhHOST, tvhHTTP, tvhUsername, tvhPassword)
		self.__hostName = tvhHOST
		self.__httpPort = tvhHTTP
		self.__username = tvhUsername
		self.__password = tvhPassword
		self.__logger = logger

		#self.__logger("Logging test")

	#Demo function to show that the init worked.
	def showServer(self):
		print("Showing stuff:", self.__hostName, self.__httpPort, self.__username, self.__password)


	#Read some JSON from TVH and convert the result into a Python dict object.
	#http://<host>:<port>/api/<url>
	def __readTVH(self, url):
		tvhResponse = []
		#TODO - Does not trap 'connection refused'
		try:
			with request.urlopen("http://" + self.__hostName + ":" + str(self.__httpPort) + "/api/" + url) as response:
				tmpJSON = response.read()
				tmpJSON=tmpJSON.decode('utf-8','ignore').encode("utf-8")  #Clean invalid UTF8 bytes.
			tvhResponse = json.loads(str(tmpJSON, 'utf8'))
		except urllib.error.HTTPError as e:
			print("Error: ", e)
		#print("Entries:", str(len(tvhResponse['entries'])))
		if 'entries' in tvhResponse:
			return tvhResponse['entries']
		else:
			return tvhResponse


	#Create a TVH Object
	def __createOBJ(self, url, payload):
		tvhResponse = ""
		# data = json.dumps(payload)
		# data = parse.urlencode(payload).encode()
		# #print("urlencode", type(data), data)
		# data = data.decode("utf-8")
		# #print("STR", type(data), data)

		newObject = "class=" + payload['class'] + "&conf=" + json.dumps(payload['conf'], separators=(',', ':'))
		newObject = newObject.replace(" ", "%20")

		#print("NEWOBJ", type(newObject), newObject)

		json_url = "http://" + self.__hostName + ":" + str(self.__httpPort) + "/api/" + url + "?" + newObject
		#print("URL ", type(json_url), json_url)
		
		#print("URL ", type(json_url), json_url)
		req = request.Request(json_url, method="GET")
		req.add_header('Content-Type', 'application/x-www-form-urlencoded; charset=UTF-8')
		#print("-------------")
		#print(req.get_method())
		#print(req.get_full_url())
		#print(req.header_items())
		#print("-------------")

		r = request.urlopen(req)
		tvhResponse = r.read()
		#print(tvhResponse)
		jRet = json.loads(tvhResponse)
		uu = ""
		if 'uuid' in jRet:
			uu = jRet['uuid']
		else:
			uu = str(tvhResponse)
		extra = ""
		#TODO, the calling function should pass this information.
		if 'name' in payload['conf']:
			extra = " name = '" + payload['conf']['name'] + "'"
		if 'networkname' in payload['conf']:
			extra = " networkname = '" + payload['conf']['networkname'] + "'"
		if 'display_label' in payload['conf']:
			extra = " display_label = '" + payload['conf']['display_label'] + "'"
		self.__logger("Created object, class '" + payload['class'] + "' uuid = '" + uu + "'" + extra)

		return tvhResponse


	#Update a TVH Object
	def __updateOBJ(self, url, object):
		tvhResponse = {}
		#print("updating object")
		#updatedObject = json.dumps(object, separators=(',', ':'))
		#updatedObject = updatedObject.replace(" ", "%20")
		#updatedObject = str.encode(updatedObject)
		#print(json.dumps(object, separators=(',', ':')))

		updatedObject = json.dumps(object, separators=(',', ':'))
		updatedObject = updatedObject.replace(" ", "%20")

		json_url = "http://" + self.__hostName + ":" + str(self.__httpPort) + "/api/" + url + "?node=" + updatedObject
		#print(json_url)

		try:
			with urllib.request.urlopen(json_url) as f:
				tvhResponse = f.read().decode('utf-8')
		except urllib.error.HTTPError as e:
			print("Error: ", e)
			tvhResponse = e

		#print("Returning", tvhResponse)
		return tvhResponse

	def getNetworkType(self, networkType):
		ret = networkType
		if networkType in self.__networkTypes:
			ret = self.__networkTypes[networkType]['scanfile']
		return ret

	#Get a list of channels from TVH
	#/channel/grid
	def getChannels(self):
		return self.__readTVH("channel/grid")


	#Get and arbitraty object using its UUID.
	def getObject(self, uuid):
		return self.__readTVH("idnode/load?uuid=" + uuid)


	#Get the list of valid object classes
	def getObjectClasses(self):
		return self.__readTVH("classes")


	#Set the GUI language
	def setGUILanguage(self, lang):
		newLang = {}
		newLang['language_ui'] = lang
		ret = self.__updateOBJ("config/save", newLang)
		return ret


	#Set the TVH Server Name
	def setServerName(self, name):
		newName = {}
		newName['server_name'] = name
		ret = self.__updateOBJ("config/save", newName)
		return ret


	# #Set the TVH Wizard complete status
	# def setWizard(self, state):
		# newWiz = {}
		# newWiz['wizard'] = state
		# ret = self.__updateOBJ("config/save", newWiz)
		# return ret


	#Set the Parental Rating processing flag
	def setParentalRating(self, val):
		newVal = {}
		newVal['epgdb_processparentallabels'] = val
		ret = self.__updateOBJ("epggrab/config/save", newVal)
		return ret


	#Set the Parental Rating processing flag
	def setOTAGenreTranslate(self, transl):
		newTransl = {}
		newTransl['ota_genre_translation'] = transl
		ret = self.__updateOBJ("epggrab/config/save", newTransl)
		return ret


	#Get all of the 'leaf' adapters.
	#In the development version, only DVB-T adapters were loaded.
	#This is because I only have DVB-T on my system.
	#ATSC and other systems should be easy enough to add in the future.
	def getAdapters(self):
		myAdapters = {}
		adapterCount = 0
		tmpAdapters = []

		for nets in self.__networkTypes:
			myAdapters[nets] = []
			myAdapters[nets] = self.__readTVH("idnode/load?class=" + self.__networkTypes[nets]['adapter'])
			adapterCount = adapterCount + len(myAdapters[nets])
			if len(myAdapters[nets]) == 0:
				del myAdapters[nets]
			#print(nets)

		# for x in myAdapters:
			# print(x, len(myAdapters[x]))
			# for y in myAdapters[x]:
				# print("  ", y['text'])
		return myAdapters


	#Get all of the scanfiles of a specific type.
	def getScanFiles(self, type, satPos=None):
		if satPos == None:
			ret = self.__readTVH("dvb/scanfile/list?type=" + self.__networkTypes[type]['scanfile'])
		else:
			ret = self.__readTVH("dvb/scanfile/list?type=" + self.__networkTypes[type]['scanfile'] + "&satpos=" + str(satPos * 10))
		myScanFiles = {}
		for sf in ret:
			txt = sf['val'].split(":")
			country = txt[0]
			region = txt[1]
			#print(country,"",region)
			if country not in myScanFiles:
				myScanFiles[country] = {}
				myScanFiles[country]['regions'] = []
			tmp = {}
			txt = region.split("-", 1) #Only split on the first hyphen
			if len(txt) == 1:
				tmp['name'] = region
			else:
				tmp['name'] = txt[1]
			tmp['path'] = sf['key']
			myScanFiles[country]['regions'].append(tmp)
		return myScanFiles


	#Create a new network
	def networkCreate(self, name, type, adapters, muxFile):
		#print("Creating network '" + name + "' Type: '" + self.__networkTypes[type]['network'] + "' Adapters: '" + str(adapters) + "'")
		newNetwork = {}
		newNetwork['class'] = self.__networkTypes[type]['network']
		newNetwork['conf'] = {}
		newNetwork['conf']['enabled'] = True
		newNetwork['conf']['networkname'] = name
		newNetwork['conf']['bouquet'] = False
		newNetwork['conf']['scanfile'] = muxFile
		newNetwork['conf']['pnetworkname'] = "TVH Quick Start"
		newNetwork['conf']['nid'] = 0
		newNetwork['conf']['autodiscovery'] = 1
		newNetwork['conf']['ignore_chnum'] = False
		newNetwork['conf']['satip_source'] = 0
		newNetwork['conf']['charset'] = ""
		newNetwork['conf']['skipinitscan'] = True
		newNetwork['conf']['idlescan'] = False
		newNetwork['conf']['sid_chnum'] = False
		newNetwork['conf']['localtime'] = 0

		ret = self.__createOBJ("mpegts/network/create", newNetwork)
		jRet = json.loads(ret)

		muxRet = []
		#Get a list of muxes for the newly created network.  These
		#should be pre-populated from the scan file provided suring creation.
		if 'uuid' in jRet:
			url = 'mpegts/mux/grid?filter=[{"field":"network_uuid","type":"string","value":"' + jRet['uuid'] + '","comparison":"eq"}]'
			muxRet = self.__readTVH(url)

		return jRet, muxRet


	#Link an adpater to a network
	def networkLinkAdapter(self, network, adapter):
		#print("Linking network '" + network + "' to adapter '" + adapter + "'")

		updatedAdapter = {}
		updatedAdapter['uuid'] = adapter
		updatedAdapter['enabled'] = True  #For new systems, adapters are disabled by default.
		updatedAdapter['networks'] = []

		adapterNetworks = []
		thisAdapter = self.getObject(adapter)
		#print(thisAdapter)
		#print(len(thisAdapter))
		#print(thisAdapter[0]['text'])

		if(thisAdapter):
			for param in thisAdapter[0]['params']:
				#print(param['id'])
				if (param['id'] == 'networks'):
					#print(param['value'])
					#Add the networks to the new list
					for n in param['value']:
						updatedAdapter['networks'].append(n)

		#Add the new network to the new list.
		updatedAdapter['networks'].append(network)
		#print(updatedAdapter)

		ret = self.__updateOBJ("idnode/save", updatedAdapter)

		#print("Got", ret)

		if ret != "{}":
			print("Failure!!")
		else:
			print("Success")

		#TODO - Do a return code here


	#Set off a network scan.
	def networkPerformScan(self, network):
		#print("Scanning network '" + network + "'")
		return self.__readTVH("mpegts/network/scan?uuid=" + network)


	#Get the current status of a specific network
	#Specifically, look for scans in progress, etc.
	def networkGetStatus(self, network):
	#http://10.1.1.252:9981/api/mpegts/mux/grid

	#filter=[{"field":"network_uuid","type":"string","value":"5fde3ce6d6231a85038b5710211af12b","comparison":"eq"}]
	#http://10.1.1.252:9981/api/mpegts/mux/grid?filter=[{%22field%22:%22network_uuid%22,%22type%22:%22string%22,%22value%22:%225fde3ce6d6231a85038b5710211af12b%22,%22comparison%22:%22eq%22}]

		#Get the current mux status.
		url = 'mpegts/mux/grid?filter=[{"field":"network_uuid","type":"string","value":"' + network + '","comparison":"eq"}]'
		ret = self.__readTVH(url)

		#Get the current LCN count by mux
		url = 'service/list?enum=0&list=lcn,multiplex_uuid'
		svcRet = self.__readTVH(url)
		#print("SERVICES", len(svcRet))

		lcnFound = 0
		lcnMux = {}
		failedMux = []
		for svc in svcRet:
			for param in svc['params']:
				if param['id'] == "multiplex_uuid":
					tmpMux = param['value']
				if param['id'] == "lcn":
					tmpLCN = param['value']
			if tmpMux not in lcnMux:
				lcnMux[tmpMux] = {}
				lcnMux[tmpMux]['blankCount'] = 0
				lcnMux[tmpMux]['count'] = 0
				lcnMux[tmpMux]['flag'] = False
			if tmpLCN == 0:
				lcnMux[tmpMux]['blankCount'] = lcnMux[tmpMux]['blankCount'] + 1
			else:
				lcnMux[tmpMux]['count'] = lcnMux[tmpMux]['count'] + 1
		#Eventually a scan will start automatically.
		#Mux Scan states
		#0 = Idle, #1 = Pending
		#scan results
		#0 = Not run yet,  #1 = OK,  #2 = Fail
		response = {}
		response['remaining'] = len(ret)
		response['serviceCount'] = 0
		response['lcns'] = 0
		#print("LCNMUX", lcnMux)
		#print(len(ret))
		for mux in ret:
			#print(mux['frequency']/1000000, "state", mux['scan_state'], "result", mux['scan_result'], "num", mux['num_svc'])
			#if mux['scan_result'] == 2:
			#Some scans sort of half worked.  They would have a failed status, yet
			#still have a TSID and ONID.  Only fail if these fields are not found
			#as well as a failed status.
			if mux['scan_result'] == 2 and mux['tsid'] == 65536  and mux['onid'] == 65536:
				failedMux.append(mux['uuid'])
			if mux['uuid'] in lcnMux:
				response['lcns'] = response['lcns'] + lcnMux[mux['uuid']]['count']
				lcnMux[mux['uuid']]['flag'] = True
			if mux['scan_state'] == 0:
				response['remaining'] = response['remaining'] - 1
			response['serviceCount'] = response['serviceCount'] + mux['num_svc']
		return response, lcnMux, failedMux


	#Map all services to channels for a given network
	#Need to firstly get all of the muxes on a network and
	#then get all of the services on that mux.
	def networkMapServices(self, network, createTags):
		chList = []
		tagList = []
		url = 'mpegts/mux/grid?limit=999999999&filter=[{"field":"network_uuid","type":"string","value":"' + network + '","comparison":"eq"}]'
		muxes = self.__readTVH(url)

		for mux in muxes:
			#In the mux, the 'pnetwork_name' is the name of the broadcaster.
			#Create a channel tag for each of them.
			tagUUID = ""
			if createTags and (mux['num_svc'] != 0):
				newTag = {}
				newTag['class'] = "channeltag"
				newTag['conf'] = {}
				newTag['conf']['enabled'] = True
				newTag['conf']['internal'] = False
				newTag['conf']['private'] = False
				newTag['conf']['index'] = 0
				#Use the mux name as default.
				if 'name' in mux:
					newTag['conf']['name'] = mux['name']
				#Is the provide name is also supplied, use that instead.
				if 'pnetwork_name' in mux:
					newTag['conf']['name'] = mux['pnetwork_name']
				ret = self.__createOBJ("channeltag/create", newTag)
				jRet = json.loads(ret)
				if 'uuid' in jRet:
					tagUUID = jRet['uuid']
					tagList.append(jRet['uuid'])

			#print(mux['name'])
			url2 = 'mpegts/service/grid?limit=999999999&filter=[{"field":"multiplex_uuid","type":"string","value":"' + mux['uuid'] + '","comparison":"eq"}]'
			services = self.__readTVH(url2)
			for service in services:
				#print("    ", service['svcname'], service['lcn'])
				#If a service is found without a name, invent one.
				if "svcname" not in service:
					tmpMux = service['multiplex']
					if 'pnetwork_name' in mux:
						tmpMux = mux['pnetwork_name']
					service['svcname'] = "??" + tmpMux + "-sid-" + str(service['sid']) + " ??"
				newChannel = {}
				newChannel['class'] = "channel"
				newChannel['conf'] = {}
				newChannel['conf']['enabled'] = True
				newChannel['conf']['autoname'] = True
				newChannel['conf']['name'] = service['svcname']
				newChannel['conf']['number'] = service['lcn']
				newChannel['conf']['services'] = []
				newChannel['conf']['services'].append(service['uuid'])
				newChannel['conf']['tags'] = []
				if len(tagUUID) != 0:
					newChannel['conf']['tags'].append(tagUUID)
				ret = self.__createOBJ("channel/create", newChannel)
				#print("New Channel:", ret)
				jRet = json.loads(ret)
				if 'uuid' in jRet:
					chList.append(jRet['uuid'])
				#count the new channels and return the number
		return chList, tagList

	#Perform a rescan on an individual Mux
	#Set the 'scan state' and then let TVH do the rest.
	def muxPerformScan(self, mux):
		updatedMux = {}
		updatedMux['uuid'] = mux
		updatedMux['scan_state'] = 2 #Idle pending
		ret = self.__updateOBJ("idnode/save", updatedMux)


	#Create Rating Label object
	def ratingLabelCreate(self, label):
		newLabel = {}
		newLabel['class'] = "ratinglabel"
		newLabel['conf'] = label
		ret = self.__createOBJ("ratinglabel/create", newLabel)
		jRet = json.loads(ret)
		return jRet

